<?php

namespace WDB_ADDONS\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
} // Exit if accessed directly

class WDB_Setup_Wizard_Init {

	/**
	 * Parent Menu Page Slug
	 */
	const MENU_PAGE_SLUG = 'wdb_addons_setup_page';

	/**
	 * Menu capability
	 */
	const MENU_CAPABILITY = 'manage_options';

	/**
	 * [$_instance]
	 * @var null
	 */
	private static $_instance = null;

	/**
	 * [instance] Initializes a singleton instance
	 * @return [Woolentor_Admin_Init]
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function __construct() {
		$this->remove_all_notices();

		$this->init();
	}

	/**
	 * [init] Assets Initializes
	 * @return [void]
	 */
	public function init() {
		add_action( 'admin_menu', [ $this, 'add_menu' ], 25 );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_action( 'wp_ajax_save_setup_wizard_settings', [ $this, 'save_settings' ] );
	}

	/**
	 * [add_menu] Admin Menu
	 */
	public function add_menu() {
		add_submenu_page(
			self::MENU_PAGE_SLUG,
			esc_html__( 'DesignBox', 'designbox-builder' ),
			esc_html__( 'DesignBox', 'designbox-builder' ),
			self::MENU_CAPABILITY,
			self::MENU_PAGE_SLUG,
			[ $this, 'render_wizard' ]
		);

	}

	/**
	 * [enqueue_scripts] Add Scripts Base Menu Slug
	 *
	 * @param  [string] $hook
	 *
	 * @return [void]
	 */
	public function enqueue_scripts( $hook ) {
		if ( isset( $_GET['page'] ) && $_GET['page'] == 'wdb_addons_setup_page' ) {

			// CSS
			wp_enqueue_style( 'wdb-admin', WDB_ADDONS_URL . '/assets/css/wdb-admin.css' );

			// JS
			wp_enqueue_script( 'wdb-admin', WDB_ADDONS_URL . '/assets/js/wdb-admin.js', array( 'jquery' ), WDB_ADDONS_VERSION, true );

			$localize_data = [
				'ajaxurl'  => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'wdb_admin_nonce' ),
				'adminURL' => admin_url(),
			];
			wp_localize_script( 'wdb-admin', 'WDB_ADDONS_ADMIN', $localize_data );

		}
	}

	/**
	 * Save Settings
	 * Save EA settings data through ajax request
	 *
	 * @access public
	 * @return  void
	 * @since 1.1.2
	 */
	public function save_settings() {

		check_ajax_referer( 'wdb_admin_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'you are not allowed to do this action', 'designbox-builder' ) );
		}

		if ( ! isset( $_POST['settings'] ) ) {
			return;
		}

		if ( empty( $_POST['settings'] ) ) {
			wp_send_json( __( 'Option name not found!', 'designbox-builder' ) );
		}

		foreach ( $_POST['settings'] as $key => $setting ) {

			$option_name = $key;

			wp_parse_str( $setting, $settings );

			$settings = array_map( 'sanitize_text_field', $settings );

			$settings = array_fill_keys( array_keys( $settings ), true );

			// update new settings
			if ( ! empty( $option_name ) ) {
				$updated = update_option( $option_name, $settings );
			}
		}

		update_option( 'wdb_addons_setup_wizard', 'complete' );

		wp_send_json_success( [ 'redirect_url' => esc_url( admin_url( 'admin.php?page=wdb_addons_settings' ) ) ] );

	}

	/**
	 * get Settings tabs to admin panel.
	 *
	 * @param array $tabs Array of tabs.
	 *
	 * @return bool|true|void
	 */
	protected function get_settings_tab() {
		$settings_tab = [
			'configure'   => [
				'title'    => esc_html__( 'Configure', 'designbox-builder' ),
				'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 21.98 22.5">
                                        <path d="M21,13.4,20,12.23a1.81,1.81,0,0,1,0-1.92L21,9.1a2.88,2.88,0,0,0,1-1.59,2.89,2.89,0,0,0-.61-1.75l-.52-.89c-.45-.78-.72-1.25-1.29-1.48a2.85,2.85,0,0,0-1.95.16l-1.25.35a1.32,1.32,0,0,1-.88-.1l-.33-.18A1.43,1.43,0,0,1,14.62,3l-.35-1.05a2.82,2.82,0,0,0-.89-1.6A2.84,2.84,0,0,0,11.57,0H10.4A2.83,2.83,0,0,0,8.6.34,2.88,2.88,0,0,0,7.7,2L7.36,3a1.35,1.35,0,0,1-.5.64l-.32.19a1.37,1.37,0,0,1-.88.12L4.38,3.55a2.85,2.85,0,0,0-1.95-.16c-.57.23-.84.7-1.29,1.48l-.52.9A2.83,2.83,0,0,0,0,7.51c.06.59.4,1,1,1.59L2,10.27a1.81,1.81,0,0,1,0,1.92L1,13.41C.41,14,.07,14.4,0,15a.75.75,0,1,0,1.49.15,3.66,3.66,0,0,1,.59-.74L3.2,13.16a3.27,3.27,0,0,0,.61-1.91,3.25,3.25,0,0,0-.64-1.94L2.09,8.09a3.61,3.61,0,0,1-.59-.73,3.64,3.64,0,0,1,.42-.85l.52-.89A3.66,3.66,0,0,1,3,4.78,3.86,3.86,0,0,1,4,5l1.32.37a2.84,2.84,0,0,0,2-.26l.38-.22A2.86,2.86,0,0,0,8.78,3.47l.35-1.05a3.56,3.56,0,0,1,.34-.86,3.38,3.38,0,0,1,.93-.06h1.17a3.38,3.38,0,0,1,.93.06,3.44,3.44,0,0,1,.35.86l.36,1.08a2.88,2.88,0,0,0,1.16,1.4l.38.22a2.84,2.84,0,0,0,2,.23L18,5a3.38,3.38,0,0,1,1-.21,4,4,0,0,1,.55.84l.52.9a3.45,3.45,0,0,1,.42.84,3.86,3.86,0,0,1-.59.74L18.78,9.34a3.24,3.24,0,0,0,0,3.85l1.08,1.22a3.52,3.52,0,0,1,.59.73,3.63,3.63,0,0,1-.42.84l-.52.9a3.59,3.59,0,0,1-.56.84,4,4,0,0,1-1-.21l-1.31-.37a2.84,2.84,0,0,0-2,.26l-.38.22A2.85,2.85,0,0,0,13.2,19l-.35,1a3.12,3.12,0,0,1-.35.86,3.38,3.38,0,0,1-.93.06H10.4a3.38,3.38,0,0,1-.93-.06,3.62,3.62,0,0,1-.34-.86.75.75,0,0,0-1.43.47,2.91,2.91,0,0,0,.9,1.61,2.83,2.83,0,0,0,1.8.34h1.17a2.84,2.84,0,0,0,1.81-.34,2.85,2.85,0,0,0,.89-1.61l.34-1a1.38,1.38,0,0,1,.5-.63l.33-.19a1.36,1.36,0,0,1,.88-.12L17.6,19a2.85,2.85,0,0,0,1.95.16c.57-.23.84-.7,1.29-1.48l.52-.9A2.85,2.85,0,0,0,22,15C21.91,14.4,21.57,14,21,13.4Z"/>
                                        <path d="M10,16.49l.06,0h.52a17.45,17.45,0,0,0,2.64-.14,3.45,3.45,0,0,0,2-.9,3.67,3.67,0,0,0,.62-.78,3,3,0,0,0,.32-1,31.41,31.41,0,0,0,.1-3.72h0a1.16,1.16,0,0,0-2-.82l-2,2a.52.52,0,0,1-.73-.73l2-2a1.16,1.16,0,0,0-.82-2h0A31.5,31.5,0,0,0,9,6.4a3,3,0,0,0-1,.32,3.67,3.67,0,0,0-.78.62,3.51,3.51,0,0,0-.9,2A17.45,17.45,0,0,0,6.08,12c0,.21,0,.33,0,.42s0,.07,0,.08h0s0,0,0,.06l-.3.3L.82,17.75a2.78,2.78,0,0,0,3.93,3.93h0l-.35-.35-.71-.71a1.28,1.28,0,0,1-1.81-1.81l4.89-4.9a2.65,2.65,0,0,0,.68-.86A2.86,2.86,0,0,0,7.58,12a15.84,15.84,0,0,1,.12-2.4A1.88,1.88,0,0,1,8.62,8a1.66,1.66,0,0,1,.55-.16,23.42,23.42,0,0,1,2.68-.08L10.48,9.17A2,2,0,0,0,13.33,12l1.37-1.37a23.32,23.32,0,0,1-.08,2.68,1.66,1.66,0,0,1-.16.55,2.09,2.09,0,0,1-.36.44,2,2,0,0,1-1.17.48,15.84,15.84,0,0,1-2.4.12,2.69,2.69,0,0,0-1.08.14,2.5,2.5,0,0,0-.86.67l-4.9,4.89,1.06,1.06,4.9-4.89ZM12.67,7.8V7.05h0Z"/>
                                    </svg>',
				'callback' => 'render_configure_tab',
			],
			'integration' => [
				'title'    => esc_html__( 'Integration', 'designbox-builder' ),
				'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 22.5 22.5">
                                        <path d="M12.12,22.5a.76.76,0,0,1-.71-1,2.36,2.36,0,0,0,.14-.8,2.4,2.4,0,0,0-4.8,0,2.36,2.36,0,0,0,.14.8.76.76,0,0,1-.71,1h0c-2.43,0-4-.07-5-1.14S0,18.75,0,16.33a.75.75,0,0,1,.32-.62.78.78,0,0,1,.68-.1,2.21,2.21,0,0,0,.8.14,2.4,2.4,0,0,0,0-4.8,2.36,2.36,0,0,0-.8.14A.75.75,0,0,1,.32,11,.75.75,0,0,1,0,10.38c0-2.42.07-4,1.14-5,.93-.93,2.21-1.1,4.12-1.13,0-.1,0-.21,0-.31a3.9,3.9,0,0,1,7.8,0c0,.1,0,.21,0,.31,1.9,0,3.19.2,4.12,1.13s1.1,2.22,1.13,4.12h.31a3.9,3.9,0,0,1,0,7.8h-.31c0,1.91-.2,3.19-1.13,4.12s-2.61,1.13-5,1.14Zm-3-5.7a3.91,3.91,0,0,1,3.9,3.9A2.81,2.81,0,0,1,13,21c1.63,0,2.53-.16,3.06-.69s.69-1.75.7-4a.76.76,0,0,1,1-.71,2.21,2.21,0,0,0,.8.14,2.4,2.4,0,0,0,0-4.8,2.45,2.45,0,0,0-.8.14.75.75,0,0,1-.68-.1.74.74,0,0,1-.32-.61c0-2.23-.08-3.35-.7-4s-1.75-.69-4-.7a.74.74,0,0,1-.61-.32.75.75,0,0,1-.1-.68,2.45,2.45,0,0,0,.14-.8,2.4,2.4,0,0,0-4.8,0,2.36,2.36,0,0,0,.14.8.75.75,0,0,1-.1.68.74.74,0,0,1-.61.32c-2.23,0-3.36.08-4,.7s-.66,1.43-.69,3.06A3.93,3.93,0,0,1,5.7,13.35a3.91,3.91,0,0,1-3.9,3.9H1.51c0,1.64.16,2.53.69,3.06S3.62,21,5.26,21a2.81,2.81,0,0,1,0-.29A3.91,3.91,0,0,1,9.15,16.8Z" />
                                    </svg>',
				'callback' => 'render_integration_tab',
			],
			'widgets'     => [
				'title'    => esc_html__( 'Widgets', 'designbox-builder' ),
				'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 22.5 22.5">
                                        <path d="M20.69,3.26C19.54,1.43,19,.52,18,.18a2.84,2.84,0,0,0-2,0c-.94.34-1.51,1.25-2.67,3.08l.64.4-.64-.4c-1.29,2-1.93,3.07-1.79,4.11a2.91,2.91,0,0,0,1,1.85c.81.68,2,.68,4.44.68s3.62,0,4.44-.68a2.93,2.93,0,0,0,1-1.85C22.62,6.32,22,5.3,20.69,3.26ZM20.5,8.07C20.11,8.4,19,8.4,17,8.4s-3.09,0-3.48-.33a1.4,1.4,0,0,1-.49-.9c-.07-.51.54-1.48,1.57-3.11h0c.91-1.44,1.46-2.31,1.91-2.47a1.35,1.35,0,0,1,1,0c.46.16,1,1,1.91,2.47,1,1.62,1.64,2.6,1.58,3.11A1.46,1.46,0,0,1,20.5,8.07Z" />
                                        <path d="M1.27,9.31C2.08,9.9,3.1,9.9,5,9.9s2.87,0,3.68-.59a3.13,3.13,0,0,0,.68-.68C9.9,7.82,9.9,6.8,9.9,5s0-2.87-.59-3.68A3.13,3.13,0,0,0,8.63.59C7.82,0,6.8,0,5,0S2.08,0,1.27.59a3.13,3.13,0,0,0-.68.68C0,2.08,0,3.1,0,5S0,7.82.59,8.63A3.13,3.13,0,0,0,1.27,9.31ZM1.8,2.15a1.93,1.93,0,0,1,.35-.35c.42-.3,1.27-.3,2.8-.3s2.38,0,2.8.3a1.93,1.93,0,0,1,.35.35c.3.42.3,1.27.3,2.8s0,2.38-.3,2.8a1.93,1.93,0,0,1-.35.35c-.42.3-1.26.3-2.8.3s-2.38,0-2.8-.3a1.93,1.93,0,0,1-.35-.35c-.3-.42-.3-1.26-.3-2.8S1.5,2.57,1.8,2.15Z" />
                                        <path d="M17,12.6a5,5,0,1,0,4.95,5A5,5,0,0,0,17,12.6ZM17,21a3.45,3.45,0,1,1,3.45-3.45A3.46,3.46,0,0,1,17,21Z" />
                                        <path d="M9.15,16.8H5.7V13.35a.75.75,0,0,0-1.5,0V16.8H.75a.75.75,0,0,0,0,1.5H4.2v3.45a.75.75,0,0,0,1.5,0V18.3H9.15a.75.75,0,1,0,0-1.5Z" />
                                    </svg>',
				'callback' => 'render_widgets_tab',
			],
			'extensions'  => [
				'title'    => esc_html__( 'Extensions', 'designbox-builder' ),
				'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 22.5 22.5">
                                        <path d="M21.15,19a5.09,5.09,0,1,1,0-6.1.75.75,0,0,0,1.2-.9,6.53,6.53,0,0,0-5.27-2.64l-.57,0A6.55,6.55,0,0,0,15.2.15a.75.75,0,0,0-.9,1.2,5.08,5.08,0,1,1-6.1,0A.75.75,0,0,0,7.3.15,6.57,6.57,0,0,0,6,9.36l-.57,0A6.53,6.53,0,0,0,.15,12a.75.75,0,1,0,1.2.9,5,5,0,0,1,4.07-2A5.09,5.09,0,1,1,1.35,19a.75.75,0,1,0-1.2.9A6.58,6.58,0,0,0,11.25,19a6.58,6.58,0,0,0,11.1.91.75.75,0,1,0-1.2-.9ZM10.67,12a5.63,5.63,0,0,0,.58,0,5.49,5.49,0,0,0,.58,0,6.67,6.67,0,0,0-.58.91A6.67,6.67,0,0,0,10.67,12Z" />
                                    </svg>',
				'callback' => 'render_extensions_tab',
			],
			'congrats'    => [
				'title'    => esc_html__( 'Congrats', 'designbox-builder' ),
				'icon'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 22.5 22.5">
                                        <path d="M7.84,2.11a9.87,9.87,0,0,1,6.82,0,.86.86,0,0,0,.27.05.75.75,0,0,0,.7-.49.74.74,0,0,0-.44-1,11.25,11.25,0,0,0-7.88,0,.75.75,0,1,0,.53,1.4Z" />
                                        <path d="M21.75,10.5a.75.75,0,0,0-.75.75,9.75,9.75,0,0,1-19.5,0,.75.75,0,0,0-1.5,0,11.25,11.25,0,0,0,22.5,0A.75.75,0,0,0,21.75,10.5Z" />
                                        <path d="M3.31,8.28l-.2.9a1.39,1.39,0,0,0,.42,1.55,1.38,1.38,0,0,0,1.6-.07L6,10.14l.87.52a1.82,1.82,0,0,0,.91.29,1.14,1.14,0,0,0,.69-.22,1.39,1.39,0,0,0,.42-1.55l-.21-.9.73-.74a1.36,1.36,0,0,0,.43-1.41,1.38,1.38,0,0,0-1.17-.9l-.93-.15L7.22,4A1.4,1.4,0,0,0,6,3.15H6A1.39,1.39,0,0,0,4.78,4L4.26,5.08l-.94.15a1.38,1.38,0,0,0-1.16.9,1.37,1.37,0,0,0,.42,1.41ZM4.58,6.54a1.44,1.44,0,0,0,1-.74L6,4.94l.42.86a1.45,1.45,0,0,0,1,.74l.74.13-.59.6a1.43,1.43,0,0,0-.36,1.27l.15.67-.66-.4A1.4,1.4,0,0,0,6,8.63a1.43,1.43,0,0,0-.7.18l-.66.4.15-.67a1.43,1.43,0,0,0-.36-1.27l-.6-.6Z" />
                                        <path d="M15.28,4l-.52,1.06-.94.15a1.38,1.38,0,0,0-1.16.9,1.37,1.37,0,0,0,.42,1.41l.73.74-.2.9a1.28,1.28,0,0,0,2,1.48l.87-.52.87.52a1.82,1.82,0,0,0,.91.29,1.14,1.14,0,0,0,.69-.22,1.39,1.39,0,0,0,.42-1.55l-.21-.9.73-.74a1.36,1.36,0,0,0,.43-1.41,1.38,1.38,0,0,0-1.17-.9l-.93-.15L17.72,4a1.4,1.4,0,0,0-1.22-.87h0A1.39,1.39,0,0,0,15.28,4Zm.79,1.78.43-.86.42.86a1.45,1.45,0,0,0,1,.74l.74.13-.59.6a1.43,1.43,0,0,0-.36,1.27l.15.67-.66-.4a1.4,1.4,0,0,0-.7-.18,1.43,1.43,0,0,0-.7.18l-.66.4.15-.67a1.43,1.43,0,0,0-.36-1.27l-.6-.6.75-.13A1.44,1.44,0,0,0,16.07,5.8Z" />
                                        <path d="M14.85,14a4.5,4.5,0,0,1-7.2,0,.75.75,0,1,0-1.2.9,6,6,0,0,0,9.6,0,.75.75,0,0,0-1.2-.9Z" />
                                    </svg>',
				'callback' => 'render_congrats_tab',
			],
		];

		return apply_filters( 'wdb_wizard_settings_tabs', $settings_tab );
	}

	protected function get_setup_config() {
		$config = [
			'widgets'      => [
				'general-elements' => [
					'title'    => __( 'General Widgets', 'designbox-builder' ),
					'elements' => [
						'image-box'      => [
							'label' => esc_html__( 'Image Box', 'designbox-builder' ),
						],
						'typewriter'     => [
							'label' => esc_html__( 'Typewriter', 'designbox-builder' ),
						],
						'animated-title' => [
							'label' => esc_html__( 'Animated Title', 'designbox-builder' ),
						],
						'animated-text'  => [
							'label' => esc_html__( 'Animated Text', 'designbox-builder' ),
						],
						'social-icons'   => [
							'label' => esc_html__( 'Social Icons', 'designbox-builder' ),
						],
						'image'          => [
							'label' => esc_html__( 'Image', 'designbox-builder' ),
						],
						'image-gallery'  => [
							'label' => esc_html__( 'Image Gallery', 'designbox-builder' ),
						],
						'counter'        => [
							'label' => esc_html__( 'Counter', 'designbox-builder' ),
						],
						'icon-box'       => [
							'label' => esc_html__( 'Icon Box', 'designbox-builder' ),
						],
						'testimonial'    => [
							'label' => esc_html__( 'Testimonial', 'designbox-builder' ),
						],
						'posts'          => [
							'label' => esc_html__( 'Posts', 'designbox-builder' ),
						],
						'button'         => [
							'label' => esc_html__( 'Button', 'designbox-builder' ),
						],
						'progressbar'    => [
							'label' => esc_html__( 'Progressbar', 'designbox-builder' ),
						],
						'team'           => [
							'label' => esc_html__( 'Team', 'designbox-builder' ),
						],
						'timeline'       => [
							'label' => esc_html__( 'Timeline', 'designbox-builder' ),
						],
						'tabs'           => [
							'label' => esc_html__( 'Tabs', 'designbox-builder' ),
						],
					]
				],
			],
			'extensions'   => [
				'custom-css'   => [
					'label' => esc_html__( 'Custom CSS', 'designbox-builder' ),
				],
				'wrapper-link' => [
					'label' => esc_html__( 'Wrapper Link', 'designbox-builder' ),
				],
				'popup'        => [
					'label' => esc_html__( 'Popup', 'designbox-builder' ),
				],
			],
			
		];

		return apply_filters( 'wdb_wizard_setup_config', $config );
	}

	/**
	 * Render wizard
	 * @return [void]
	 */
	public function render_wizard() {
		?>
        <div class="wrap wdb-admin-wrapper">
            <div class="wizard-form">
				<?php
				$tabs = $this->get_settings_tab();
				//svg allowed
				$kses_defaults = wp_kses_allowed_html( 'post' );
				$svg_args      = array(
					'svg'   => array(
						'class'           => true,
						'aria-hidden'     => true,
						'aria-labelledby' => true,
						'role'            => true,
						'xmlns'           => true,
						'width'           => true,
						'height'          => true,
						'viewbox'         => true // <= Must be lower case!
					),
					'g'     => array( 'fill' => true ),
					'title' => array( 'title' => true ),
					'path'  => array(
						'd'    => true,
						'fill' => true
					)
				);
				$allowed_svg   = array_merge( $kses_defaults, $svg_args );

				if ( ! empty( $tabs ) ) {
					?>
                    <div class="wizard-wrap">
                        <div class="wizard">
                            <div class="wizard-bar" style="--width: 0;" data-wizard-bar></div>
                            <ul class="wizard-list">
								<?php
								foreach ( $tabs as $key => $el ) {
									?>
                                    <li class="wizard-item" data-wizard-item>
                                        <div class="icon"><?php echo wp_kses( $el['icon'], $allowed_svg ); ?></div>
                                        <div class="title"><?php echo esc_html( $el['title'] ); ?></div>
                                        <div class="step-circle"></div>
                                    </li>
									<?php
								}
								?>
                            </ul>
                        </div>
                    </div>
					<?php
					//tab content
					foreach ( $tabs as $key => $el ) {
						if ( isset( $el['callback'] ) ) {
							call_user_func( [ $this, $el['callback'] ], $key, $el );
						}
					}
				}
				?>
                <div class="wizard-buttons">
                    <button class="wdb-admin-btn" type="button" data-btn-previous="true">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 16.29 12">
                            <path d="M0,6a2,2,0,0,1,.59-1.4L4.9.29A1,1,0,0,1,5.61,0a1.05,1.05,0,0,1,.71.29,1.12,1.12,0,0,1,.22.33A1,1,0,0,1,6.61,1a1.07,1.07,0,0,1-.07.39,1.21,1.21,0,0,1-.22.32L3,5H15.28a1,1,0,0,1,1,1,1,1,0,0,1-.3.71,1,1,0,0,1-.71.3H3l3.31,3.29a1,1,0,0,1,.29.7,1,1,0,0,1-.29.71,1,1,0,0,1-1.42,0L.59,7.41A2,2,0,0,1,0,6Z"/>
                        </svg>
						<?php esc_html_e( 'Back', 'designbox-builder' ); ?>
                    </button>
                    <button type="button" class="wdb-admin-btn wdb-settings-next" data-btn-next="true">
						<?php esc_html_e( 'Next Step', 'designbox-builder' ); ?>
                    </button>
                </div>
            </div>
        </div>
		<?php
	}

	/**
	 * Render Configure tab
	 * @return [void]
	 */
	public function render_configure_tab() {
		?>
        <div class="wizard-content" data-form-tab>
            <div class="wizard-welcome">
                <h2><?php echo esc_html__( 'Welcome to the DesignBox Builder', 'designbox-builder' ); ?></h2>
                <div class="desc"> <?php echo esc_html__( "Elevate your website's visual appeal with seamless animations that bring your content to life. With easy integration into Elementor, designing dynamic and engaging web experiences seems simpler, smarter, and quicker. Get ready to mesmerize your audience and stand out from the crowd with the DesignBox Builder", 'designbox-builder' ); ?></div>
            </div>
        </div>
		<?php
	}

	

	/**
	 * Render Widget tab
	 * @return [void]
	 */
	public function render_widgets_tab() {
		?>
        <div class="wizard-content" data-form-tab>
            <form action="POST" class="wdb-settings" name="wdb_save_widgets">
				<?php $widgets = $this->get_setup_config()['widgets']; ?>
				<?php foreach ( $widgets as $group ) { ?>
                    <div class="settings-group">
                        <div class="title-area">
                            <h4><?php echo esc_html( $group['title'] ); ?></h4>
                        </div>
                        <div class="settings-wrapper">
							<?php foreach ( $group['elements'] as $key => $widget ) { ?>
                                <div class="item">
                                    <div class="title"><?php echo esc_html( $widget['label'] ); ?></div>
                                    <div class="actions">
                                        <div class="switcher">
                                            <input type="checkbox" class="wdb-settings-item"
                                                   id="<?php echo esc_attr( $key ); ?>"
                                                   name="<?php echo esc_attr( $key ); ?>" checked="checked">
                                            <label for="<?php echo esc_attr( $key ); ?>">
                                                <span class="control-label-switch" data-active="ON"
                                                      data-inactive="OFF"></span>
                                            </label>
                                        </div>
                                    </div>
                                </div>
							<?php } ?>
                        </div>
                    </div>
				<?php } ?>
            </form>
        </div>
		<?php
	}

	/**
	 * Render Extension tab
	 * @return [void]
	 */
	public function render_extensions_tab() {
		?>
        <div class="wizard-content" data-form-tab>
            <form action="POST" class="wdb-settings" name="wdb_save_extensions">
                <div class="settings-group">
                    <div class="settings-wrapper">
						<?php $extensions = $this->get_setup_config()['extensions']; ?>
						<?php foreach ( $extensions as $key => $extension ) { ?>
                            <div class="item">
                                <div class="title"><?php echo esc_html( $extension['label'] ); ?></div>
                                <div class="actions">
                                    <div class="switcher">
                                        <input type="checkbox" class="wdb-settings-item"
                                               id="<?php echo esc_attr( $key ); ?>"
                                               name="<?php echo esc_attr( $key ); ?>" checked="checked">
                                        <label for="<?php echo esc_attr( $key ); ?>">
                                                    <span class="control-label-switch" data-active="ON"
                                                          data-inactive="OFF"></span>
                                        </label>
                                    </div>
                                </div>
                            </div>
						<?php } ?>
                    </div>
                </div>
            </form>
        </div>
		<?php
	}

	/**
	 * Render Congrats tab
	 * @return [void]
	 */
	public function render_congrats_tab() {
		?>
        <div class="wizard-content" data-form-tab>
            <div class="wizard-congrats">
                <div class="icon">
                    <svg xmlns="http://www.w3.org/2000/svg" width="264" height="210" viewBox="0 0 264 210" fill="none">
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M244.92 124.471C244.867 124.269 244.891 124.055 244.987 123.87C245.023 123.809 245.046 123.741 245.054 123.67C245.137 123.32 245.403 123.136 245.886 123.136C246.063 123.133 246.24 123.155 246.41 123.203C246.544 123.253 246.694 123.295 246.843 123.337C247.006 123.384 247.174 123.407 247.343 123.403C247.637 123.38 247.922 123.289 248.175 123.136C248.702 122.83 249.175 122.438 249.574 121.977C249.95 121.543 250.285 121.074 250.573 120.577C250.818 120.149 251.029 119.703 251.205 119.242C251.364 118.817 251.48 118.5 251.572 118.275C251.884 117.154 251.907 115.972 251.638 114.84C251.387 113.616 251.007 112.423 250.506 111.279C249.947 109.937 249.278 108.643 248.508 107.41C247.94 106.495 247.598 105.457 247.509 104.383C247.425 103.282 247.538 102.175 247.842 101.114C248.177 99.978 248.659 98.8903 249.274 97.8788C249.92 96.8114 250.667 95.8089 251.505 94.8852C252.609 93.5147 254.006 92.4092 255.593 91.6498C256.642 91.1871 257.776 90.9486 258.922 90.9494C260.017 90.9879 261.079 91.3347 261.986 91.95C262.711 92.3362 263.294 92.9437 263.651 93.6844C263.931 94.2885 264.045 94.956 263.984 95.619C263.931 96.2124 263.737 96.7844 263.418 97.2867C263.208 97.682 262.886 98.0068 262.494 98.2207C262.242 98.3339 261.97 98.3924 261.695 98.3924C261.419 98.3924 261.147 98.3339 260.895 98.2207C260.704 98.0816 260.526 97.9253 260.363 97.7537C260.096 97.4452 259.63 97.3701 258.964 97.5202C258.191 97.7284 257.459 98.0668 256.8 98.5209C255.962 99.0842 255.193 99.7447 254.51 100.489C253.764 101.27 253.156 102.172 252.712 103.157C252.232 104.199 252.092 105.366 252.313 106.493C252.701 107.915 253.357 109.251 254.244 110.428C254.625 111.051 254.967 111.698 255.268 112.363C255.668 113.204 256.013 114.07 256.3 114.956C256.619 115.91 256.842 116.893 256.966 117.891C257.102 118.892 257.056 119.909 256.833 120.893C256.441 122.716 255.489 124.369 254.111 125.621C252.959 126.601 251.588 127.287 250.115 127.623C249.04 127.889 247.905 127.759 246.918 127.256C246.417 126.968 245.969 126.596 245.595 126.155C245.229 125.674 245.019 125.092 244.995 124.487L244.92 124.471ZM153.899 104.458C154.032 104.141 154.182 103.774 154.331 103.357C154.481 102.94 154.631 102.523 154.764 102.123C154.898 101.723 155.122 101.164 155.297 100.722C155.427 100.389 155.619 100.083 155.863 99.8217C156.004 99.6577 156.188 99.5364 156.394 99.4716C156.6 99.4068 156.82 99.4009 157.029 99.4548C157.384 99.5841 157.729 99.7402 158.061 99.9217C158.524 100.185 158.931 100.536 159.26 100.956C159.654 101.462 159.917 102.058 160.026 102.69C160.144 103.533 160.053 104.393 159.759 105.192C159.626 105.592 159.434 106.126 159.193 106.793C158.952 107.46 158.702 108.152 158.46 108.861C158.219 109.57 157.969 110.237 157.728 110.862C157.486 111.487 157.295 111.996 157.162 112.396C156.412 114.363 155.832 116.392 155.43 118.459C155.206 119.876 155.272 120.702 155.63 120.96C155.691 121.013 155.758 121.058 155.83 121.094C155.88 121.144 155.925 121.2 155.963 121.26C156.003 121.319 156.048 121.375 156.096 121.427C156.183 121.59 156.228 121.772 156.228 121.957C156.228 122.141 156.183 122.323 156.096 122.486C155.882 122.755 155.615 122.977 155.311 123.137C155.007 123.298 154.674 123.394 154.331 123.42C153.913 123.516 153.479 123.517 153.061 123.425C152.642 123.333 152.248 123.149 151.909 122.886C151.688 122.757 151.505 122.574 151.376 122.353C151.246 122.144 151.134 121.923 151.043 121.694C150.904 121.463 150.792 121.217 150.71 120.96C150.583 120.624 150.483 120.278 150.411 119.926C150.344 119.593 150.286 119.292 150.244 119.026C150.197 118.75 150.197 118.468 150.244 118.192C150.279 117.349 150.39 116.512 150.577 115.69C149.601 117.15 148.411 118.454 147.047 119.559C146.518 120.03 145.894 120.381 145.218 120.588C144.541 120.795 143.828 120.854 143.126 120.76C142.018 120.629 140.992 120.109 140.23 119.292C139.604 118.564 139.139 117.712 138.864 116.791C138.549 115.709 138.414 114.582 138.465 113.455C138.511 112.676 138.6 111.899 138.731 111.129C138.864 110.295 138.997 109.62 139.131 108.994C139.305 108.235 139.489 107.502 139.663 106.793C140.102 105.375 140.59 103.944 141.129 102.498C141.614 101.159 142.193 99.8547 142.86 98.5959C143.376 97.6084 143.951 96.6532 144.583 95.7358C145.239 94.7617 146.2 94.0341 147.314 93.6678C148.018 93.3969 148.792 93.3734 149.511 93.6011C149.819 93.6608 150.109 93.7873 150.362 93.9717C150.616 94.1561 150.825 94.394 150.977 94.6684C151.068 94.9388 151.068 95.2319 150.977 95.5023C150.943 95.6118 150.872 95.7062 150.777 95.7691C150.155 96.4306 149.618 97.1673 149.179 97.9622C148.687 98.796 148.113 99.9634 147.447 101.431C146.781 102.899 146.073 104.766 145.316 107.026C145.011 107.827 144.733 108.627 144.483 109.428C144.267 110.095 144.051 110.804 143.851 111.562C143.668 112.24 143.524 112.928 143.418 113.622C143.259 114.423 143.206 115.242 143.26 116.057C143.283 116.49 143.405 116.913 143.618 117.291C143.675 117.423 143.768 117.536 143.885 117.618C144.003 117.7 144.141 117.749 144.284 117.758C144.997 117.684 145.671 117.393 146.215 116.924C146.958 116.314 147.629 115.62 148.213 114.856C148.908 113.969 149.542 113.036 150.111 112.063C150.71 111.037 151.26 110.036 151.776 109.061C152.292 108.085 152.716 107.193 153.066 106.393L153.899 104.6V104.458ZM142.244 119.384C141.574 118.525 141.118 117.519 140.912 116.449C140.665 115.155 140.565 113.838 140.612 112.521C140.711 110.496 141.103 108.496 141.778 106.584C142.603 103.845 143.578 101.153 144.7 98.5209C145.072 97.6112 145.494 96.7232 145.965 95.8608C145.277 96.5962 144.694 97.4241 144.234 98.3207C143.751 99.2129 143.185 100.397 142.569 101.89C141.802 103.718 141.168 105.599 140.671 107.518C140.122 109.358 139.842 111.268 139.838 113.188C139.835 114.42 140.026 115.644 140.404 116.816C140.631 117.541 141.005 118.211 141.503 118.784C141.903 119.184 142.18 119.384 142.336 119.384H142.244ZM154.548 116.883C154.906 115.215 155.289 113.714 155.713 112.321C156.138 110.929 156.546 109.711 156.945 108.686C157.437 107.485 157.894 106.401 158.344 105.417C157.477 106.579 156.782 107.859 156.279 109.219C155.825 110.437 155.447 111.682 155.147 112.947C154.846 114.241 154.645 115.557 154.548 116.883ZM180.113 74.5139C179.592 74.3655 179.056 74.276 178.515 74.2471C177.869 74.2052 177.221 74.2614 176.592 74.4139C175.899 74.584 175.251 74.9026 174.694 75.3478C174.035 75.8925 173.52 76.5913 173.195 77.3824C172.887 78.0495 172.363 79.3087 171.53 81.1765C170.698 83.0444 169.765 85.2875 168.708 87.9141C167.651 90.5408 166.527 93.4093 165.378 96.5363C164.229 99.6632 163.156 102.84 162.182 106.034C161.832 107.101 161.482 108.269 161.149 109.536C160.816 110.804 160.542 112.038 160.317 113.163C160.072 114.304 159.938 115.466 159.917 116.632C159.872 117.078 159.892 117.527 159.976 117.967C160.067 118.367 160.175 118.8 160.317 119.267C160.447 119.735 160.697 120.16 161.041 120.501C161.224 120.678 161.425 120.834 161.641 120.968L162.34 121.235C162.634 121.318 162.945 121.318 163.239 121.235C163.531 121.181 163.807 121.063 164.049 120.89C164.291 120.718 164.491 120.494 164.637 120.235C164.724 120.094 164.769 119.933 164.769 119.768C164.769 119.603 164.724 119.441 164.637 119.301L164.371 119.034L164.238 118.9C163.799 118.634 163.91 117.21 164.571 114.631C164.746 114.056 164.929 113.455 165.104 112.838C165.278 112.221 165.478 111.571 165.703 110.904C166.258 109.036 166.946 106.99 167.767 104.766C168.6 102.548 169.432 100.247 170.423 97.8704C171.414 95.4939 172.463 93.0757 173.62 90.6075C174.777 88.1393 176.017 85.7127 177.341 83.3112C177.636 82.7761 177.97 82.2633 178.34 81.7769C178.656 81.2933 179.031 80.7346 179.472 80.1092C179.913 79.4838 180.429 78.825 181.004 78.1162C181.123 77.9899 181.234 77.8562 181.337 77.716C181.563 77.4454 181.722 77.1255 181.803 76.7821C181.85 76.5436 181.845 76.2976 181.787 76.0614C181.73 75.8251 181.622 75.6044 181.47 75.4145C181.11 75.0053 180.651 74.6953 180.138 74.5139H180.113ZM165.453 101.348C164.787 103.082 164.21 104.769 163.722 106.409C163.231 108.077 162.814 109.52 162.456 110.812C162.057 112.321 161.707 113.764 161.391 115.14C161.467 113.874 161.645 112.616 161.924 111.379C162.24 109.939 162.639 108.405 163.122 106.776C163.613 105.158 164.163 103.491 164.787 101.773C165.412 100.055 166.028 98.4375 166.652 96.7697C168.117 93.0007 169.757 89.0259 171.572 84.8455C170.906 86.7134 170.073 88.9565 169.074 91.5748C168.084 94.1931 166.868 97.4118 165.453 101.306V101.348ZM188.071 104.416C188.123 105.217 188.01 106.021 187.738 106.776C187.472 107.51 187.097 108.385 186.606 109.411C185.807 111.014 185.163 112.689 184.683 114.414C184.417 115.573 184.458 116.257 184.816 116.482C184.871 116.473 184.928 116.473 184.983 116.482C185.038 116.493 185.094 116.493 185.149 116.482C185.271 116.473 185.394 116.473 185.516 116.482C185.723 116.535 185.915 116.638 186.073 116.782C186.177 116.843 186.266 116.924 186.335 117.022C186.404 117.12 186.451 117.232 186.473 117.349C186.489 117.499 186.473 117.651 186.428 117.795C186.382 117.938 186.306 118.071 186.207 118.183C185.841 118.534 185.398 118.793 184.913 118.938C184.428 119.083 183.915 119.11 183.418 119.017C182.514 118.855 181.679 118.425 181.02 117.783C180.491 117.163 180.165 116.394 180.088 115.582C180.005 114.748 179.955 114.072 179.955 113.58C179.156 114.431 178.323 115.265 177.457 116.082C176.625 116.916 175.792 117.683 174.96 118.35C174.217 118.963 173.413 119.5 172.562 119.951C171.942 120.308 171.246 120.514 170.531 120.551C169.947 120.575 169.366 120.45 168.844 120.187C168.321 119.925 167.873 119.534 167.543 119.051C166.801 117.942 166.471 116.608 166.61 115.282C166.677 113.735 166.98 112.208 167.509 110.754C168.081 109.112 168.803 107.526 169.665 106.017C170.549 104.474 171.539 102.994 172.629 101.589C173.632 100.284 174.746 99.0672 175.959 97.9538C176.958 97.0306 178.079 96.2479 179.289 95.6274C180.135 95.1375 181.114 94.9274 182.086 95.027C183.21 95.1773 184.27 95.6362 185.149 96.3528C185.821 96.9124 186.354 97.6203 186.706 98.4208C186.98 99.0143 187.084 99.6727 187.006 100.322C186.923 100.881 186.673 101.156 186.273 101.223C186.025 101.281 185.764 101.245 185.541 101.123C185.418 101.017 185.328 100.878 185.283 100.722C185.233 100.564 185.191 100.414 185.149 100.255C185.098 100.091 185.006 99.9419 184.883 99.8217C184.797 99.7467 184.696 99.6902 184.587 99.6557C184.479 99.6213 184.364 99.6097 184.25 99.6215C183.863 99.6475 183.488 99.7616 183.151 99.9551C182.572 100.276 182.036 100.669 181.553 101.123C180.771 101.841 180.037 102.61 179.355 103.424C178.378 104.582 177.479 105.805 176.666 107.085C175.706 108.552 174.874 109.986 174.169 111.387C173.534 112.546 172.978 113.747 172.504 114.981C172.146 115.982 172.146 116.507 172.504 116.549C172.729 116.591 173.129 116.416 173.703 116.015C174.404 115.503 175.072 114.946 175.701 114.348C176.45 113.664 177.258 112.872 178.123 111.988C178.989 111.104 179.838 110.187 180.621 109.253C181.403 108.37 182.136 107.446 182.818 106.484C183.375 105.712 183.876 104.901 184.317 104.058C184.344 103.87 184.344 103.679 184.317 103.491C184.303 103.302 184.303 103.112 184.317 102.924C184.292 102.872 184.279 102.815 184.279 102.757C184.279 102.699 184.292 102.642 184.317 102.59C184.317 102.365 184.467 102.165 184.783 101.99C185.107 101.85 185.454 101.771 185.807 101.756C186.177 101.727 186.548 101.809 186.873 101.99C187.225 102.18 187.515 102.47 187.705 102.824C187.949 103.264 188.086 103.755 188.105 104.258L188.071 104.416ZM168.309 117.274C168.132 116.282 168.132 115.265 168.309 114.273C168.578 112.945 168.957 111.643 169.441 110.378C170.231 108.389 171.212 106.482 172.371 104.683C173.413 102.847 174.701 101.164 176.2 99.6799C177.154 98.7379 178.239 97.9401 179.422 97.3117C180.113 96.9245 180.895 96.7288 181.686 96.7447C182.261 96.8114 182.594 96.9365 182.685 97.1116C181.673 97.2461 180.698 97.5866 179.822 98.1122C178.723 98.7643 177.708 99.5484 176.8 100.447C175.425 101.823 174.228 103.367 173.237 105.042C172.164 106.731 171.207 108.492 170.373 110.312C169.815 111.548 169.326 112.815 168.908 114.106C168.588 115.147 168.386 116.221 168.309 117.308V117.274ZM205.969 80.6845C206.622 79.6543 207.381 78.6957 208.234 77.8244C208.357 77.7515 208.469 77.6617 208.567 77.5575C208.794 77.3151 208.975 77.0323 209.099 76.7237C209.24 76.3243 209.24 75.8889 209.099 75.4896C208.999 75.2544 208.848 75.0446 208.656 74.8758C208.465 74.7069 208.238 74.5833 207.992 74.5139C207.471 74.3655 206.935 74.276 206.394 74.2471C205.745 74.2051 205.094 74.2613 204.463 74.4139C203.763 74.5869 203.108 74.9051 202.54 75.3478C201.862 75.8822 201.335 76.5832 201.008 77.3824C200.783 77.8661 200.442 78.6332 199.976 79.6756C199.509 80.7179 198.943 82.0187 198.311 83.578C198.211 83.5667 198.111 83.5667 198.011 83.578C197.889 83.5915 197.766 83.5915 197.645 83.578C196.415 83.6593 195.215 83.9883 194.115 84.5453C193.287 84.9893 192.527 85.5511 191.859 86.213C191.231 86.8291 190.713 87.5482 190.327 88.3394C190.194 88.9231 190.327 89.29 190.694 89.4401C191.128 89.6076 191.597 89.6649 192.059 89.6069C192.802 89.5425 193.538 89.4085 194.257 89.2066C194.606 89.1232 194.939 89.0315 195.247 88.9398L196.179 88.6729C195.347 90.7159 194.515 92.9089 193.682 95.2438C192.85 97.5786 192.017 99.9468 191.185 102.373C190.352 104.8 189.661 107.218 189.02 109.636C188.379 112.054 187.855 114.356 187.455 116.532C187.333 117.105 187.244 117.684 187.189 118.267C187.147 118.733 187.147 119.202 187.189 119.668C187.217 120.054 187.356 120.425 187.588 120.735C187.885 121.12 188.22 121.475 188.587 121.794C188.904 122.107 189.293 122.336 189.72 122.461C189.892 122.506 190.075 122.502 190.245 122.449C190.416 122.396 190.568 122.296 190.685 122.161C191.046 121.625 191.2 120.976 191.118 120.335C191.25 119.007 191.484 117.691 191.817 116.399C192.284 114.314 192.983 111.738 193.915 108.669C194.848 105.6 196.01 102.223 197.403 98.5375C198.802 94.8519 200.411 91.1439 202.232 87.4138C204.871 86.9438 207.537 86.643 210.215 86.5132C212.696 86.3965 214.868 86.3631 216.733 86.4132C218.903 86.4576 220.92 86.5688 222.785 86.7467C223.104 86.8449 223.448 86.8212 223.751 86.68C223.926 86.5746 224.056 86.4084 224.117 86.213C224.135 86.0931 224.12 85.9704 224.073 85.8586C224.026 85.7468 223.949 85.6501 223.851 85.5793C223.603 85.3678 223.321 85.1988 223.018 85.079C222.641 84.8862 222.237 84.7513 221.819 84.6787C218.616 83.9098 215.356 83.3995 212.071 83.1528C209.558 82.9784 207.037 82.9561 204.521 83.0861C204.654 82.7775 204.804 82.4773 204.954 82.1855C205.104 81.8936 205.253 81.6435 205.387 81.4183C205.52 81.1932 205.745 80.9097 205.919 80.6845H205.969ZM205.303 75.3561C204.856 75.5938 204.433 75.873 204.038 76.19C203.657 76.4807 203.293 76.7925 202.947 77.1239C202.548 77.4978 202.201 77.9241 201.915 78.3914C202.048 77.8562 202.249 77.3404 202.515 76.8571C202.777 76.4529 203.113 76.1021 203.505 75.8231C204.035 75.4701 204.672 75.3165 205.303 75.3895V75.3561ZM191.734 105.876C191.296 107.299 190.896 108.688 190.535 110.045C190.177 111.379 189.894 112.546 189.703 113.572C189.395 114.773 189.129 115.882 188.87 116.908C189.004 115.623 189.212 114.231 189.503 112.738C189.795 111.246 190.144 109.736 190.569 108.202C190.993 106.668 191.401 105.133 191.934 103.607C192.467 102.081 192.908 100.597 193.399 99.1713C194.548 95.8859 195.833 92.5754 197.253 89.24C196.771 90.7076 196.304 92.0973 195.855 93.4093C195.405 94.7212 194.95 96.0526 194.49 97.4035C194.032 98.7377 193.574 100.097 193.133 101.473C192.691 102.849 192.217 104.316 191.734 105.867V105.876ZM199.226 103.549C199.36 103.188 199.493 102.821 199.626 102.448L199.959 101.481L200.358 100.547C200.484 100.209 200.664 99.8936 200.891 99.6132C201.017 99.4401 201.196 99.3134 201.401 99.2531C201.605 99.1927 201.824 99.2021 202.023 99.2797C202.399 99.368 202.758 99.5143 203.089 99.7133C203.557 99.98 203.971 100.33 204.313 100.747C204.719 101.246 204.984 101.845 205.079 102.482C205.197 103.325 205.106 104.184 204.812 104.983C204.504 105.817 204.18 106.651 203.847 107.485C203.514 108.319 203.197 109.153 202.889 109.986C202.133 111.929 201.553 113.935 201.158 115.982C200.933 117.408 200.999 118.25 201.357 118.484H201.424C201.596 118.572 201.735 118.712 201.824 118.884C201.91 119.048 201.955 119.232 201.955 119.417C201.955 119.603 201.91 119.786 201.824 119.951C201.645 120.208 201.405 120.414 201.124 120.551C200.78 120.732 200.409 120.856 200.026 120.918C199.618 120.993 199.201 120.993 198.793 120.918C198.371 120.828 197.973 120.646 197.628 120.385C197.433 120.277 197.263 120.129 197.129 119.951C196.978 119.757 196.855 119.544 196.762 119.317C196.626 119.085 196.515 118.839 196.429 118.584C196.299 118.237 196.198 117.88 196.13 117.516C196.063 117.158 196.013 116.849 195.963 116.582C195.925 116.306 195.925 116.025 195.963 115.748C195.983 114.853 196.095 113.961 196.296 113.088C196.521 112.063 196.787 111.004 197.129 109.92C197.47 108.836 197.786 107.727 198.161 106.584C198.535 105.442 198.902 104.458 199.26 103.524L199.226 103.549ZM199.693 106.609C199.332 107.677 199.032 108.711 198.793 109.711C198.552 110.712 198.336 111.613 198.161 112.413C197.986 113.339 197.803 114.231 197.628 115.073C197.47 113.757 197.571 112.422 197.928 111.146C198.302 109.586 198.76 108.102 199.226 106.676C199.841 104.885 200.553 103.128 201.357 101.414C201.163 102.204 200.93 102.983 200.658 103.749C200.367 104.541 200.05 105.5 199.693 106.609ZM209.474 90.0154C209.839 90.1886 210.134 90.4833 210.306 90.8493C210.436 91.1436 210.459 91.4736 210.373 91.7832C210.307 92.046 210.194 92.2946 210.04 92.517C209.959 92.6342 209.87 92.7457 209.774 92.8506C209.164 93.5373 208.628 94.2863 208.175 95.0853L207.176 96.8448C207.085 96.9782 206.968 97.1783 206.81 97.4452C206.649 97.7205 206.46 97.9777 206.244 98.2123C206.034 98.4513 205.787 98.6542 205.511 98.8127C205.387 98.8858 205.249 98.9325 205.106 98.9497C204.962 98.9669 204.817 98.9543 204.679 98.9128C203.97 98.5807 203.354 98.0762 202.889 97.4452C202.727 97.1997 202.667 96.9004 202.723 96.6113C202.744 96.2562 202.785 95.9026 202.848 95.5523C202.911 95.2259 202.992 94.9031 203.089 94.585C203.136 94.3897 203.203 94.1998 203.289 94.018C203.346 93.7355 203.421 93.4569 203.514 93.1841C203.58 93.009 203.663 92.8256 203.747 92.6505C203.969 91.9653 204.358 91.3464 204.879 90.8493C205.334 90.4291 205.865 90.1007 206.444 89.882C206.975 89.6835 207.543 89.6038 208.109 89.6485C208.59 89.6846 209.062 89.7972 209.507 89.9821L209.474 90.0154ZM203.647 96.1443C203.823 95.292 204.068 94.4554 204.379 93.6428C204.65 93.0097 204.985 92.4062 205.378 91.8416C205.548 91.5752 205.772 91.3481 206.036 91.1755C206.3 91.0029 206.598 90.8889 206.91 90.841C206.51 91.2829 206.078 91.7749 205.711 92.3086C205.345 92.8422 204.995 93.3592 204.612 93.9763C204.218 94.6616 203.895 95.3852 203.647 96.136V96.1443ZM218.689 93.2175C220.425 93.2128 222.103 93.8412 223.409 94.9853C224.458 96.0002 225.197 97.2929 225.541 98.7126C225.9 100.197 225.979 101.735 225.774 103.249C225.29 107.225 223.863 111.028 221.611 114.339C220.504 115.926 219.159 117.332 217.624 118.509C216.463 119.389 215.155 120.055 213.761 120.476C212.801 120.768 211.808 120.937 210.806 120.977C209.538 120.947 208.3 120.591 207.21 119.943C206.67 119.606 206.203 119.163 205.837 118.642C205.471 118.12 205.213 117.531 205.079 116.908C204.669 115.367 204.523 113.769 204.646 112.18C204.759 110.445 205.084 108.732 205.611 107.076C206.124 105.45 206.84 103.895 207.742 102.448C208.528 101.143 209.517 99.9714 210.673 98.9795C211.138 97.3214 212.1 95.8459 213.428 94.7518C214.928 93.5749 216.83 93.0358 218.723 93.2508L218.689 93.2175ZM208.575 103.482C209.175 102.414 209.952 101.455 210.873 100.647C211.541 100.078 212.263 99.5748 213.029 99.1463C213.748 98.8181 214.511 98.5935 215.293 98.4792C214.052 98.9564 212.903 99.6447 211.896 100.514C210.998 101.306 210.196 102.202 209.507 103.182C208.836 104.122 208.277 105.138 207.842 106.209C207.436 107.226 207.113 108.275 206.877 109.344C206.339 111.883 206.159 114.485 206.344 117.074C206.074 116.359 205.928 115.604 205.911 114.84C205.848 113.707 205.881 112.572 206.011 111.446C206.172 110.09 206.451 108.751 206.843 107.443C207.239 106.05 207.821 104.718 208.575 103.482ZM221.611 105.876C221.14 107.46 220.507 108.991 219.722 110.445C218.952 111.896 218.025 113.258 216.958 114.506C216.051 115.589 214.971 116.513 213.761 117.241C213.307 117.544 212.796 117.75 212.259 117.847C211.722 117.945 211.171 117.932 210.639 117.808C210.323 117.716 210.073 117.349 209.874 116.707C209.651 115.871 209.573 115.002 209.641 114.139C209.708 112.914 209.898 111.699 210.207 110.512C210.588 109.022 211.133 107.579 211.83 106.209C212.105 106.469 212.42 106.683 212.762 106.843C212.988 106.936 213.237 106.957 213.475 106.903C213.714 106.849 213.93 106.723 214.094 106.543C214.184 106.452 214.252 106.341 214.292 106.22C214.333 106.098 214.345 105.969 214.327 105.842C214.3 105.524 214.233 105.21 214.127 104.908C214.026 104.603 213.904 104.304 213.761 104.016C213.663 103.809 213.552 103.609 213.428 103.416C214.015 102.426 214.885 101.636 215.926 101.148C216.431 100.935 216.99 100.889 217.524 101.014C217.885 101.197 218.23 101.409 218.556 101.648C218.736 101.769 218.943 101.841 219.159 101.859C219.375 101.876 219.591 101.838 219.788 101.748C219.942 101.687 220.074 101.581 220.167 101.444C220.26 101.308 220.311 101.146 220.313 100.981C220.342 100.533 220.226 100.088 219.98 99.7133C219.656 99.1851 219.215 98.7397 218.689 98.4125C217.964 97.9808 217.16 97.6974 216.325 97.5786C217.124 96.7392 218.343 96.5168 219.98 96.9115C220.706 97.1898 221.317 97.7073 221.711 98.3791C222.127 99.2203 222.354 100.142 222.377 101.081C222.406 102.72 222.158 104.352 221.645 105.909L221.611 105.876ZM242.24 111.146C242.639 110.036 243.072 108.852 243.605 107.61C244.138 106.368 244.571 105.208 244.97 104.149C245.68 102.502 246.084 100.739 246.161 98.9461C246.188 98.0159 245.961 97.096 245.503 96.2861C245.273 95.9623 244.975 95.6929 244.63 95.497C244.285 95.301 243.901 95.1833 243.505 95.152C242.557 95.1333 241.626 95.4101 240.841 95.9442C239.826 96.5918 238.888 97.3527 238.044 98.2123C237.07 99.1999 236.171 100.259 235.355 101.381C234.523 102.557 233.69 103.749 232.925 104.942C233.461 103.388 234.083 101.865 234.789 100.38C235.251 99.4099 235.763 98.4636 236.321 97.5452C236.709 96.8726 237.176 96.2484 237.711 95.6857C237.761 95.657 237.804 95.6186 237.838 95.5727C237.873 95.5269 237.897 95.4746 237.911 95.4189C238.011 95.15 238.011 94.8539 237.911 94.585C237.762 94.3101 237.554 94.0717 237.303 93.8871C237.051 93.7025 236.761 93.5763 236.454 93.5177C235.735 93.29 234.96 93.3135 234.257 93.5844C233.142 93.9507 232.182 94.6783 231.526 95.6524C230.895 96.5717 230.316 97.5267 229.794 98.5125C229.155 99.7411 228.599 101.011 228.13 102.315C227.613 103.688 227.147 105.078 226.731 106.484C226.315 107.891 225.948 109.225 225.632 110.487C225.324 111.754 225.074 112.805 224.9 113.647C224.725 114.489 224.658 115.015 224.567 115.615C224.475 116.215 224.408 116.782 224.367 117.283C224.328 117.492 224.328 117.707 224.367 117.916C224.395 118.227 224.474 118.532 224.6 118.817C224.867 119.219 225.192 119.579 225.566 119.884C226.113 120.285 226.736 120.569 227.397 120.718C227.609 120.798 227.839 120.821 228.063 120.785C228.296 120.724 228.507 120.597 228.671 120.42C228.834 120.242 228.944 120.022 228.987 119.784C229.033 119.61 229.056 119.431 229.054 119.251C229.058 119.049 229.08 118.848 229.12 118.65C229.17 118.475 229.403 117.758 229.819 116.516C230.236 115.273 230.868 113.747 231.684 111.954C232.614 109.956 233.647 108.007 234.781 106.117C236.043 104.011 237.537 102.053 239.235 100.28C239.501 100.055 239.726 99.855 239.901 99.6799C240.068 99.5539 240.247 99.4423 240.433 99.3464C240.463 99.3229 240.498 99.3059 240.535 99.2964C240.572 99.287 240.61 99.2854 240.648 99.2917C240.685 99.2979 240.721 99.312 240.753 99.3329C240.785 99.3538 240.812 99.3811 240.833 99.4131C241.008 99.6382 240.999 100.055 240.833 100.681C240.493 101.656 240.104 102.613 239.668 103.549C238.903 105.214 238.246 106.927 237.703 108.677C236.989 110.939 236.468 113.257 236.146 115.607C236.059 116.315 236.015 117.028 236.013 117.741C236.012 118.358 236.079 118.974 236.213 119.576C236.336 120.178 236.586 120.746 236.945 121.244C237.378 121.789 237.955 122.202 238.61 122.436C239.34 122.749 240.161 122.772 240.908 122.503C241.216 122.419 241.332 122.194 241.241 121.844C241.198 121.658 241.131 121.479 241.041 121.31C240.861 121.037 240.737 120.731 240.675 120.41C240.608 120.076 240.55 119.776 240.508 119.509C240.466 119.2 240.443 118.888 240.442 118.575C240.483 117.606 240.628 116.645 240.875 115.707C241.261 114.148 241.74 112.614 242.306 111.112L242.24 111.146ZM228.088 106.876C228.501 104.883 229.089 102.929 229.844 101.039C230.346 99.819 230.936 98.6373 231.609 97.5035C232.101 96.6698 232.758 95.946 233.541 95.3772C232.752 96.4378 232.072 97.5747 231.509 98.771C230.871 100.105 230.316 101.373 229.844 102.573C229.289 103.991 228.715 105.434 228.121 106.901L228.088 106.876ZM243.072 104.216C243.605 102.832 243.988 101.715 244.238 100.814C244.483 99.8785 244.54 98.9032 244.404 97.9455C244.608 98.1839 244.745 98.4713 244.804 98.7794C244.898 99.1611 244.943 99.5535 244.937 99.9468C244.907 100.57 244.795 101.187 244.604 101.781C244.116 103.332 243.583 104.775 243.006 106.109C242.466 107.348 241.999 108.618 241.607 109.911C241.616 109.394 241.683 108.879 241.807 108.377C241.898 107.885 242.04 107.301 242.24 106.609C242.44 105.917 242.714 105.133 243.072 104.249V104.216ZM37.7363 70.2863C39.4217 69.8803 41.1817 69.9033 42.856 70.353C43.4543 70.534 44.0173 70.816 44.5209 71.1868C45.0732 71.6111 45.518 72.1595 45.8196 72.7878C46.1816 73.5726 46.3744 74.425 46.3856 75.2894C46.3986 76.5143 46.2417 77.7351 45.9194 78.9167C45.1786 82.32 43.184 85.3184 40.3336 87.3137C39.379 88.1164 38.1829 88.5745 36.9372 88.6146C36.4723 88.6447 36.0116 88.5112 35.6346 88.2371C35.2576 87.963 34.9881 87.5656 34.8727 87.1136C34.7034 86.5639 34.7388 85.9714 34.9726 85.4459C35.2146 84.8922 35.5161 84.3666 35.8716 83.8782C36.3375 83.3239 36.85 82.8106 37.4033 82.3439C38.7224 81.2506 39.8775 79.9728 40.8331 78.5498C41.7163 77.2584 42.1467 75.7099 42.0568 74.147C41.9736 72.8546 41.3492 72.2792 40.2004 72.4126C38.603 72.7327 37.0928 73.3925 35.7717 74.3472C33.6534 75.7495 31.6442 77.3105 29.7613 79.0168C27.4777 81.0587 25.302 83.2186 23.2431 85.4876C21.0898 87.8391 19.0586 90.2823 17.1494 92.8172C15.4009 95.1055 13.8445 97.5348 12.496 100.08C9.43251 105.989 7.43737 111.109 6.51056 115.44C5.58374 119.77 5.48108 123.362 6.20255 126.213C6.73556 128.813 8.22038 131.118 10.3649 132.676C12.5617 134.165 15.1312 135.009 17.7821 135.111C21.4665 135.38 25.1523 134.608 28.4211 132.884C31.6375 131.155 34.4852 128.812 36.804 125.988C37.0288 125.721 37.2452 125.43 37.47 125.154C37.6947 124.879 37.9361 124.587 38.2025 124.254C38.4689 123.92 38.7353 123.578 39.035 123.22C39.3852 122.784 39.7753 122.382 40.2004 122.019C40.575 121.711 40.8747 121.736 41.0912 122.086C41.2502 122.329 41.3562 122.604 41.4021 122.891C41.448 123.179 41.4328 123.472 41.3576 123.754C41.2286 124.363 41.0302 124.957 40.7665 125.521C40.4235 126.235 39.9989 126.907 39.5012 127.523C38.3528 128.951 37.1301 130.317 35.8383 131.617C34.3473 133.117 32.6683 134.418 30.8435 135.486C28.8447 136.657 26.6929 137.544 24.4502 138.121C21.9285 138.752 19.3117 138.907 16.7332 138.58C13.5344 138.36 10.4161 137.477 7.57609 135.986C5.43747 134.811 3.63047 133.113 2.32323 131.05C1.12295 129.087 0.382915 126.878 0.158839 124.587C-0.0822987 122.215 -0.0487816 119.823 0.258731 117.458C0.974651 111.815 3.0586 105.55 6.51056 98.6626C7.99044 95.7966 9.68231 93.0455 11.572 90.4324C13.2157 88.1732 14.9803 86.0047 16.8581 83.9366C18.4761 82.143 20.2235 80.4707 22.086 78.9334C23.8175 77.5409 25.4158 76.3234 27.0059 75.3061C30.3387 73.0978 33.9879 71.4107 37.8279 70.3029L37.7363 70.2863ZM7.05996 103.883C8.76225 100.156 10.8256 96.6066 13.2202 93.2842C17.5152 87.2532 22.4808 81.7309 28.0214 76.8237C24.9414 79.0881 22.049 81.5979 19.3722 84.3285C18.1734 85.5293 16.9302 86.8718 15.6427 88.3561C14.3551 89.8403 13.0815 91.4525 11.8217 93.1925C10.5162 94.9865 9.31541 96.8547 8.22543 98.7877C7.08171 100.817 6.05065 102.907 5.13697 105.05C3.63173 108.96 2.58144 113.031 2.00689 117.183C1.78385 118.841 1.69481 120.514 1.74054 122.186C1.7758 123.832 2.01058 125.467 2.43979 127.056C2.85944 128.6 3.5348 130.063 4.43772 131.383C5.39708 132.762 6.6585 133.903 8.12554 134.719C6.00534 132.887 4.45871 130.481 3.67182 127.789C3.24972 126.433 2.97071 125.035 2.83935 123.62C2.65272 121.838 2.65272 120.041 2.83935 118.258C3.08497 116.049 3.51944 113.864 4.138 111.729C4.91615 109.05 5.89286 106.433 7.05996 103.899V103.883ZM52.5709 93.2175C50.6803 93.0011 48.7807 93.5404 47.2847 94.7184C45.9543 95.8122 44.9899 97.2874 44.5209 98.9461C43.3675 99.9403 42.3786 101.111 41.5907 102.415C40.694 103.863 39.9809 105.418 39.4678 107.043C38.9362 108.697 38.6118 110.412 38.5022 112.146C38.3751 113.735 38.5215 115.335 38.9351 116.874C39.068 117.497 39.3243 118.086 39.689 118.608C40.0536 119.129 40.5191 119.572 41.0579 119.909C42.1483 120.557 43.3867 120.913 44.6541 120.943C45.6592 120.903 46.655 120.735 47.6177 120.443C49.0088 120.021 50.3139 119.355 51.472 118.475C53.0121 117.302 54.3602 115.895 55.4679 114.306C57.7158 110.993 59.1427 107.191 59.6302 103.216C59.8351 101.702 59.756 100.164 59.3971 98.6793C59.0566 97.2604 58.3204 95.9677 57.2743 94.9519C55.965 93.8066 54.2841 93.1782 52.5459 93.1841L52.5709 93.2175ZM42.4564 103.482C41.7087 104.72 41.1292 106.052 40.7332 107.443C40.3406 108.751 40.0621 110.09 39.9007 111.446C39.7713 112.572 39.7379 113.707 39.8008 114.84C39.8174 115.604 39.9638 116.359 40.2337 117.074C40.0463 114.486 40.2229 111.884 40.7582 109.344C40.998 108.276 41.3209 107.227 41.7238 106.209C42.1585 105.138 42.7173 104.122 43.3888 103.182C44.0785 102.2 44.8834 101.304 45.7863 100.514C46.793 99.6447 47.9421 98.9564 49.1827 98.4792C48.4002 98.5927 47.6377 98.8173 46.9184 99.1463C46.1497 99.5745 45.4248 100.077 44.754 100.647C43.8382 101.46 43.062 102.418 42.4564 103.482ZM55.5011 105.876C56.0068 104.317 56.2543 102.686 56.2337 101.047C56.2038 100.11 55.9769 99.1894 55.5677 98.3457C55.1714 97.6757 54.5615 97.1588 53.8362 96.8781C52.1713 96.4779 50.9504 96.7002 50.1734 97.5452C51.0091 97.6625 51.8129 97.946 52.5376 98.3791C53.065 98.7063 53.5095 99.1516 53.8362 99.6799C54.0782 100.057 54.1948 100.5 54.1692 100.947C54.1608 101.112 54.1065 101.272 54.0123 101.407C53.9181 101.543 53.7879 101.649 53.6364 101.715C53.4397 101.805 53.2231 101.843 53.0074 101.825C52.7917 101.808 52.5841 101.735 52.4044 101.614C52.0792 101.374 51.7338 101.162 51.3721 100.981C50.8386 100.856 50.2793 100.902 49.7738 101.114C48.7487 101.609 47.8938 102.399 47.318 103.382C47.4013 103.516 47.5178 103.716 47.651 103.983C47.7904 104.273 47.9127 104.571 48.0173 104.875C48.1185 105.178 48.1855 105.491 48.2171 105.809C48.2345 105.936 48.2226 106.065 48.1822 106.186C48.1417 106.308 48.0739 106.418 47.984 106.509C47.8195 106.69 47.6035 106.816 47.3652 106.87C47.127 106.923 46.878 106.902 46.652 106.809C46.3096 106.649 45.9948 106.435 45.7197 106.176C45.005 107.543 44.4465 108.986 44.0547 110.478C43.746 111.666 43.5563 112.881 43.4887 114.106C43.425 114.969 43.5038 115.837 43.7218 116.674C43.9216 117.316 44.1796 117.683 44.4876 117.775C45.0205 117.898 45.5728 117.911 46.1109 117.814C46.649 117.716 47.1617 117.51 47.6177 117.208C48.8264 116.482 49.9043 115.557 50.8061 114.473C51.876 113.227 52.8031 111.865 53.5698 110.412C54.3576 108.958 54.9936 107.427 55.4679 105.842L55.5011 105.876ZM76.1213 111.146C75.5529 112.646 75.0775 114.181 74.6978 115.74C74.4514 116.678 74.3063 117.64 74.2649 118.609C74.2626 118.921 74.2848 119.233 74.3315 119.543C74.3731 119.809 74.4314 120.11 74.498 120.443C74.5573 120.765 74.682 121.072 74.8643 121.344C74.9467 121.515 75.0136 121.694 75.0641 121.877C75.1473 122.228 75.0641 122.453 74.7311 122.536C73.9844 122.806 73.1634 122.782 72.4335 122.469C71.78 122.232 71.2041 121.82 70.7686 121.277C70.409 120.779 70.1593 120.211 70.036 119.609C69.9025 119.007 69.8355 118.392 69.8362 117.775C69.8378 117.061 69.8822 116.348 69.9694 115.64C70.294 113.29 70.8175 110.972 71.5344 108.711C72.0742 106.959 72.7305 105.246 73.499 103.582C74.0734 102.29 74.4647 101.331 74.6645 100.714C74.8643 100.097 74.8726 99.6716 74.6645 99.4464C74.6437 99.4144 74.6166 99.3871 74.5847 99.3662C74.5528 99.3453 74.5169 99.3313 74.4794 99.325C74.4418 99.3188 74.4033 99.3204 74.3664 99.3298C74.3295 99.3392 74.2949 99.3562 74.2649 99.3797C74.0772 99.4739 73.8988 99.5855 73.7321 99.7133C73.549 99.8884 73.3326 100.089 73.0662 100.314C71.3655 102.086 69.8688 104.044 68.6041 106.151C67.4697 108.039 66.4387 109.987 65.5157 111.988C64.8154 113.475 64.193 114.997 63.651 116.549C63.2264 117.791 62.9933 118.509 62.9517 118.684C62.9114 118.881 62.8891 119.082 62.8851 119.284C62.883 119.464 62.8607 119.643 62.8185 119.818C62.7741 120.056 62.6627 120.277 62.4976 120.454C62.3325 120.631 62.1204 120.758 61.8862 120.818C61.6622 120.855 61.4325 120.832 61.2202 120.752C60.5596 120.601 59.9365 120.317 59.3888 119.918C59.0178 119.61 58.6926 119.25 58.4231 118.85C58.2971 118.565 58.2184 118.261 58.19 117.95C58.1557 117.74 58.1557 117.526 58.19 117.316C58.24 116.76 58.3066 116.204 58.3898 115.648C58.4814 115.048 58.5896 114.398 58.7228 113.68C58.856 112.963 59.1473 111.788 59.4553 110.52C59.7634 109.253 60.138 107.918 60.5542 106.518C60.9704 105.117 61.4449 103.732 61.9527 102.348C62.4184 101.043 62.9747 99.7728 63.6177 98.5459C64.1365 97.5583 64.7148 96.6031 65.3492 95.6857C66.0031 94.7136 66.9605 93.9863 68.0714 93.6177C68.7753 93.3469 69.5501 93.3234 70.2691 93.551C70.5768 93.6096 70.8677 93.7358 71.121 93.9203C71.3744 94.1048 71.5837 94.3432 71.7342 94.6184C71.8344 94.8873 71.8344 95.1833 71.7342 95.4522C71.7206 95.5079 71.6959 95.5603 71.6616 95.6061C71.6272 95.652 71.584 95.6904 71.5344 95.7191C70.999 96.2834 70.5297 96.9073 70.1359 97.5786C69.6947 98.2957 69.1785 99.2463 68.6041 100.414C67.9017 101.9 67.2792 103.422 66.7394 104.975C67.497 103.783 68.3045 102.59 69.1703 101.414C69.9891 100.292 70.8907 99.233 71.8674 98.2457C72.7249 97.3713 73.6802 96.5991 74.7145 95.9442C75.4955 95.4092 76.424 95.1323 77.37 95.152C77.7656 95.1833 78.1495 95.301 78.4946 95.497C78.8398 95.6929 79.1379 95.9623 79.368 96.2861C79.833 97.0933 80.0636 98.0146 80.0339 98.9461C79.9491 100.739 79.5431 102.501 78.8352 104.149C78.4356 105.208 78.0027 106.368 77.4699 107.61C76.9371 108.852 76.5042 110.036 76.1047 111.146H76.1213ZM61.9694 106.876C62.5438 105.409 63.1265 103.966 63.701 102.548C64.1838 101.348 64.7388 100.08 65.3659 98.746C65.9256 97.55 66.6033 96.4131 67.3888 95.3522C66.6072 95.9194 65.9523 96.6436 65.4658 97.4785C64.7888 98.6105 64.1988 99.7924 63.701 101.014C62.942 102.904 62.3514 104.857 61.9361 106.851L61.9694 106.876ZM76.9538 104.216C76.6041 105.1 76.3211 105.884 76.1213 106.576C75.9215 107.268 75.78 107.852 75.6884 108.344C75.5648 108.846 75.4978 109.361 75.4886 109.878C75.8806 108.584 76.3476 107.315 76.8872 106.076C77.4988 104.664 78.0324 103.219 78.4855 101.748C78.6806 101.155 78.7926 100.537 78.8185 99.9134C78.8293 99.5199 78.7845 99.1269 78.6853 98.746C78.6269 98.4379 78.4891 98.1505 78.2857 97.9121C78.4215 98.8699 78.3649 99.8451 78.1193 100.781C77.8778 101.673 77.4866 102.799 76.9538 104.183V104.216ZM114.481 130.066C114.412 129.502 114.231 128.958 113.949 128.465C113.404 127.542 112.622 126.782 111.684 126.263C110.557 125.644 109.349 125.184 108.096 124.896C106.629 124.548 105.137 124.314 103.634 124.195C101.973 124.062 100.305 124.028 98.6395 124.095C96.883 124.162 95.1432 124.312 93.4117 124.529C94.2108 122.861 94.9323 121.219 95.576 119.601C96.2198 117.983 96.7748 116.449 97.241 114.998C97.7072 113.58 98.1067 112.305 98.4397 111.171C98.7727 110.036 99.0058 109.136 99.139 108.469C99.2613 107.922 99.2951 107.359 99.2389 106.801C99.1988 106.492 99.1205 106.19 99.0058 105.901C98.8767 105.571 98.6794 105.273 98.4268 105.026C98.1743 104.779 97.8722 104.588 97.5406 104.466C97.2712 104.367 96.9843 104.325 96.6978 104.342C96.4112 104.359 96.1313 104.436 95.8757 104.566C95.6295 104.724 95.4422 104.958 95.343 105.233C95.3015 105.364 95.3015 105.504 95.343 105.634V106.034C94.6694 107.586 93.7471 109.017 92.6125 110.27C91.4796 111.565 90.2344 112.756 88.8914 113.831C87.8027 114.732 86.6058 115.494 85.3284 116.099C84.2462 116.591 83.5219 116.699 83.1723 116.432C82.8226 116.165 82.9226 115.598 83.2389 114.531C83.6246 113.289 84.1004 112.077 84.6624 110.904C85.3118 109.528 86.0609 108.094 86.9267 106.601C87.703 105.253 88.5592 103.952 89.4907 102.707C90.1637 101.791 90.8979 100.922 91.6884 100.105C92.1636 99.5987 92.6973 99.1508 93.2784 98.771C93.6023 98.5462 93.9836 98.4189 94.3773 98.4041C94.6091 98.4047 94.8332 98.4874 95.01 98.6376C95.1463 98.7743 95.239 98.9484 95.2764 99.1379C95.3263 99.338 95.3679 99.5298 95.4095 99.7049C95.451 99.8738 95.5442 100.025 95.6759 100.139C95.8989 100.263 96.1604 100.299 96.4085 100.239C96.8081 100.197 97.0495 99.8467 97.1411 99.2046C97.2207 98.4682 97.1178 97.7236 96.8414 97.0366C96.4988 96.1542 95.9641 95.3595 95.2764 94.7101C94.4375 93.9063 93.3641 93.392 92.2129 93.2425C91.2521 93.1496 90.2917 93.4236 89.524 94.0097C88.3839 94.7749 87.3532 95.6923 86.4605 96.7364C85.3291 98.0474 84.2968 99.441 83.3721 100.906C81.3899 103.967 79.759 107.242 78.5105 110.67C77.9891 112.071 77.6098 113.521 77.3784 114.998C77.2455 115.806 77.2681 116.633 77.445 117.433C77.6106 118.141 77.8919 118.817 78.2774 119.434C78.6145 119.965 79.0705 120.411 79.6094 120.735C80.1308 121.044 80.7509 121.139 81.3409 121.002C83.4335 120.617 85.4116 119.762 87.1265 118.5C88.7437 117.33 90.2587 116.025 91.6551 114.598C90.7616 116.999 89.9291 119.123 89.1577 120.968C88.3863 122.814 87.6093 124.446 86.8268 125.863C85.4857 126.196 84.1653 126.608 82.8726 127.097C81.7933 127.494 80.7559 127.996 79.7759 128.598C78.6825 129.283 77.6761 130.097 76.779 131.025C76.1669 131.668 75.6535 132.399 75.2556 133.193C74.9611 133.771 74.8124 134.412 74.8227 135.061C74.8292 135.559 74.9077 136.053 75.0557 136.528C75.0946 136.612 75.2361 136.845 75.4803 137.229C75.7635 137.635 76.1405 137.967 76.5792 138.196C76.7582 138.28 76.9491 138.337 77.1452 138.363C77.3155 138.373 77.4835 138.407 77.6447 138.463C77.7971 138.503 77.9535 138.525 78.1109 138.53C80.681 138.609 83.1867 137.716 85.1286 136.028C87.3523 134.16 89.2731 131.957 90.8226 129.499C91.755 129.407 92.7041 129.34 93.678 129.299L96.6083 129.165C98.6252 129.071 100.646 129.093 102.66 129.232C104.156 129.335 105.645 129.513 107.122 129.766C108.143 129.933 109.152 130.164 110.144 130.458C110.927 130.708 111.493 130.891 111.809 131.025L112.109 131.192C112.242 131.282 112.346 131.41 112.409 131.558L112.741 131.892C112.929 132.05 113.163 132.143 113.407 132.159C113.765 132.251 114.007 132.159 114.14 131.959C114.277 131.727 114.389 131.481 114.473 131.225C114.566 130.857 114.588 130.475 114.54 130.099L114.481 130.066ZM79.9174 118.875C79.6436 117.649 79.5984 116.383 79.7842 115.14C79.9888 113.596 80.3572 112.078 80.883 110.612C81.6693 108.444 82.6852 106.367 83.9132 104.416C85.3017 102.083 86.857 99.8537 88.5667 97.7453C89.8986 96.1721 91.2084 95.2743 92.4959 95.052C92.2197 94.8695 91.8943 94.7763 91.5635 94.7851C91.1553 94.8094 90.7575 94.9233 90.3981 95.1187C89.7777 95.4436 89.2151 95.8691 88.7332 96.3778C88.3336 96.7781 87.7092 97.5035 86.8684 98.5459C86.0276 99.5882 85.1536 100.814 84.2379 102.215C83.3222 103.616 82.4481 105.125 81.5823 106.743C80.7618 108.265 80.0604 109.848 79.4845 111.479C78.9709 112.843 78.6895 114.283 78.652 115.74C78.6263 116.322 78.7194 116.903 78.9255 117.448C79.1316 117.993 79.4464 118.49 79.8508 118.909L79.9174 118.875ZM77.2119 135.136C77.941 134.004 78.921 133.057 80.0755 132.367C81.2958 131.667 82.5917 131.107 83.9382 130.7C83.2427 131.749 82.4512 132.732 81.574 133.635C81.0212 134.202 80.3802 134.675 79.676 135.036C79.2243 135.275 78.7216 135.401 78.2108 135.403C77.8656 135.392 77.526 135.313 77.2119 135.169V135.136ZM95.243 125.538C97.2942 125.273 99.3602 125.14 101.428 125.138C103.237 125.19 105.039 125.368 106.823 125.671C108.557 125.931 110.214 126.569 111.676 127.539C110.506 127.019 109.28 126.638 108.021 126.405C106.348 126.084 104.658 125.85 102.96 125.705C100.394 125.491 97.8163 125.436 95.243 125.538ZM107.955 93.9513C108.232 93.9775 108.51 93.9775 108.787 93.9513C109.011 93.9323 109.234 93.8989 109.453 93.8512C109.624 93.8045 109.801 93.782 109.978 93.7845C110.182 93.7054 110.394 93.6494 110.61 93.6177C110.95 93.539 111.303 93.539 111.643 93.6177C111.962 93.7023 112.265 93.8374 112.542 94.018C112.88 94.2503 113.146 94.5742 113.308 94.9519C113.61 95.4557 113.771 96.0319 113.774 96.6197C113.776 97.1817 113.709 97.7418 113.574 98.2874C113.454 98.7762 113.298 99.2555 113.108 99.7216C112.933 100.147 112.816 100.422 112.775 100.555C112.026 102.223 111.126 104.022 110.078 105.951C109.037 107.91 108.121 109.795 107.322 111.621C106.607 113.172 106.081 114.805 105.757 116.482C105.516 117.908 105.832 118.909 106.723 119.484C106.783 119.537 106.85 119.582 106.923 119.618C106.986 119.677 107.037 119.75 107.071 119.83C107.106 119.91 107.123 119.997 107.122 120.085C107.05 120.301 106.924 120.496 106.756 120.652C106.502 120.877 106.191 121.027 105.857 121.085C105.104 121.21 104.331 121.081 103.659 120.718C102.873 120.301 102.225 119.664 101.795 118.884C101.275 117.883 100.99 116.776 100.962 115.648C100.908 114.086 101.192 112.53 101.795 111.087C102.379 109.524 103.057 107.998 103.826 106.518C104.6 105.033 105.358 103.674 106.09 102.457C106.823 101.239 107.472 100.189 108.055 99.3214C108.463 98.731 108.838 98.1186 109.179 97.4869C109.31 97.3228 109.386 97.1205 109.393 96.9101C109.401 96.6997 109.34 96.4925 109.22 96.3194C109.012 96.1533 108.77 96.0344 108.511 95.9711C108.252 95.9078 107.983 95.9014 107.722 95.9525C107.34 96.0063 106.951 95.9633 106.59 95.8275C105.568 97.245 104.536 98.5764 103.493 99.8217C102.449 101.067 101.487 102.201 100.604 103.224C100.486 103.326 100.351 103.405 100.205 103.457C100.021 103.524 99.9381 103.266 99.9381 102.69C99.9573 102.087 100.07 101.491 100.271 100.922C100.488 100.23 100.671 99.7133 100.796 99.3547C100.798 99.3284 100.806 99.3028 100.819 99.2798C100.831 99.2568 100.849 99.2368 100.871 99.2213L100.995 98.9545C101.487 98.4708 101.994 97.9121 102.527 97.2867C103.06 96.6613 103.643 95.9776 104.259 95.2271C103.833 95.0461 103.458 94.7626 103.168 94.4019C102.877 94.0412 102.68 93.6143 102.594 93.1591C102.425 92.1363 102.481 91.0889 102.76 90.0905C103.41 87.7756 104.623 85.6582 106.29 83.9282C106.944 83.2337 107.738 82.6877 108.621 82.3272C108.95 82.177 109.318 82.1381 109.671 82.2163C110.024 82.2945 110.342 82.4857 110.577 82.7608C111.11 83.3364 111.427 84.0788 111.476 84.8622C111.533 85.7756 111.42 86.6917 111.143 87.5639C110.822 88.6281 110.387 89.6542 109.845 90.6242C109.29 91.6915 108.649 92.7811 107.922 93.8929L107.955 93.9513ZM106.348 92.5504C106.964 91.6582 107.514 90.7909 108.013 89.9487C108.432 89.2171 108.756 88.4354 108.979 87.6223C109.195 86.73 109.162 86.3548 108.879 86.4882C108.44 86.7781 108.084 87.1776 107.847 87.6473C107.398 88.4003 107.019 89.193 106.714 90.0154C106.38 90.804 106.265 91.6684 106.381 92.517L106.348 92.5504ZM105.149 107.56C104.708 108.494 104.317 109.394 103.917 110.262C103.518 111.129 103.218 111.929 102.952 112.563C102.685 113.197 102.352 114.114 102.119 114.823C102.353 113.062 102.812 111.339 103.484 109.695C104.108 108.138 104.831 106.623 105.649 105.158C106.524 103.628 107.503 102.16 108.579 100.764C107.955 101.965 107.355 103.115 106.781 104.233C106.207 105.35 105.674 106.451 105.183 107.568L105.149 107.56ZM131.231 104.433C131.279 105.234 131.166 106.037 130.898 106.793C130.631 107.527 130.257 108.402 129.765 109.428C128.965 111.03 128.318 112.706 127.834 114.431C127.568 115.59 127.618 116.274 127.967 116.499C128.022 116.489 128.079 116.489 128.134 116.499C128.189 116.509 128.245 116.509 128.3 116.499C128.422 116.489 128.545 116.489 128.667 116.499C128.877 116.55 129.072 116.653 129.233 116.799C129.336 116.859 129.425 116.941 129.494 117.039C129.563 117.137 129.61 117.248 129.632 117.366C129.646 117.516 129.63 117.667 129.584 117.81C129.538 117.954 129.464 118.086 129.366 118.2C128.999 118.551 128.554 118.81 128.068 118.955C127.581 119.1 127.068 119.127 126.569 119.034C125.669 118.897 124.831 118.494 124.163 117.875C123.63 117.257 123.304 116.487 123.231 115.673C123.152 115.009 123.107 114.341 123.097 113.672C122.298 114.523 121.466 115.357 120.6 116.174C119.767 117.008 118.935 117.775 118.103 118.442C117.364 119.057 116.563 119.593 115.713 120.043C115.092 120.399 114.397 120.605 113.682 120.643C113.097 120.667 112.515 120.541 111.991 120.279C111.467 120.016 111.018 119.625 110.685 119.142C109.95 118.031 109.621 116.7 109.753 115.373C109.828 113.828 110.131 112.302 110.652 110.845C111.23 109.205 111.954 107.619 112.816 106.109C113.699 104.565 114.69 103.086 115.78 101.681C116.781 100.373 117.895 99.1566 119.11 98.0456C120.111 97.1246 121.231 96.3421 122.44 95.7191C123.286 95.2292 124.265 95.0191 125.237 95.1187C126.359 95.2671 127.417 95.7263 128.292 96.4445C128.966 97.004 129.501 97.7117 129.857 98.5125C130.131 99.106 130.235 99.7644 130.157 100.414C130.065 100.972 129.824 101.248 129.424 101.314C129.176 101.374 128.915 101.338 128.692 101.214C128.565 101.11 128.473 100.971 128.425 100.814C128.425 100.656 128.342 100.505 128.292 100.347C128.244 100.181 128.152 100.031 128.026 99.9134C127.939 99.8384 127.839 99.7819 127.73 99.7475C127.621 99.7131 127.506 99.7014 127.393 99.7133C127.006 99.7415 126.631 99.8554 126.294 100.047C125.717 100.366 125.182 100.759 124.704 101.214C123.922 101.933 123.188 102.701 122.506 103.516C121.528 104.675 120.627 105.898 119.809 107.176C118.86 108.644 118.011 110.078 117.312 111.479C116.673 112.636 116.117 113.837 115.647 115.073C115.289 116.074 115.289 116.599 115.647 116.641C115.863 116.682 116.263 116.507 116.846 116.107C117.546 115.597 118.21 115.04 118.835 114.439C119.593 113.756 120.4 112.963 121.266 112.079C122.132 111.196 122.931 110.278 123.763 109.344C124.54 108.46 125.271 107.536 125.953 106.576C126.516 105.808 127.017 104.997 127.451 104.149C127.487 103.962 127.487 103.77 127.451 103.582C127.441 103.394 127.441 103.204 127.451 103.015C127.409 102.908 127.409 102.789 127.451 102.682C127.451 102.457 127.609 102.257 127.917 102.081C128.245 101.942 128.594 101.863 128.95 101.848C129.32 101.82 129.69 101.901 130.015 102.081C130.366 102.274 130.655 102.564 130.848 102.915C131.091 103.355 131.228 103.847 131.247 104.35L131.231 104.433ZM111.468 117.291C111.291 116.298 111.291 115.282 111.468 114.289C111.738 112.962 112.116 111.66 112.6 110.395C113.387 108.405 114.368 106.497 115.53 104.7C116.572 102.866 117.857 101.183 119.351 99.6966C120.309 98.756 121.397 97.9584 122.581 97.3284C123.272 96.9404 124.054 96.7446 124.846 96.7614C125.42 96.8281 125.753 96.9532 125.844 97.1283C124.831 97.2612 123.857 97.6018 122.981 98.1289C121.88 98.7806 120.862 99.5647 119.951 100.464C118.579 101.84 117.384 103.384 116.396 105.058C115.32 106.746 114.363 108.507 113.532 110.328C112.97 111.563 112.481 112.83 112.067 114.122C111.747 115.164 111.546 116.238 111.468 117.324V117.291ZM149.129 80.7012C149.781 79.671 150.54 78.7124 151.393 77.8411C151.514 77.766 151.626 77.6764 151.726 77.5742C151.951 77.3313 152.129 77.0485 152.25 76.7404C152.395 76.3418 152.395 75.9048 152.25 75.5062C152.169 75.2675 152.034 75.0505 151.857 74.8714C151.68 74.6923 151.464 74.5558 151.226 74.4722C150.705 74.3252 150.169 74.2357 149.628 74.2054C148.98 74.1631 148.328 74.2194 147.697 74.3722C146.995 74.5467 146.338 74.8647 145.765 75.3061C145.091 75.8437 144.564 76.5437 144.234 77.3407C144.017 77.8244 143.668 78.5915 143.21 79.6339C142.752 80.6762 142.177 81.977 141.545 83.5363C141.445 83.525 141.345 83.525 141.245 83.5363C141.123 83.5494 141.001 83.5494 140.879 83.5363C139.649 83.6176 138.449 83.9466 137.349 84.5036C136.519 84.9455 135.759 85.5074 135.093 86.1713C134.462 86.7846 133.944 87.5044 133.561 88.2977C133.428 88.8814 133.561 89.2483 133.928 89.3984C134.362 89.5666 134.831 89.6239 135.293 89.5652C136.034 89.5015 136.767 89.3674 137.482 89.1649C137.84 89.0815 138.173 88.9898 138.481 88.8981L139.414 88.6312C138.614 90.6714 137.782 92.8617 136.916 95.2021C136.051 97.5424 135.218 99.9189 134.419 102.332C133.586 104.75 132.895 107.176 132.254 109.595C131.613 112.013 131.089 114.314 130.689 116.491C130.563 117.062 130.474 117.642 130.423 118.225C130.381 118.691 130.381 119.16 130.423 119.626C130.444 120.014 130.583 120.387 130.823 120.693C131.12 121.079 131.454 121.434 131.822 121.752C132.138 122.065 132.527 122.294 132.954 122.419C133.127 122.465 133.309 122.46 133.479 122.407C133.65 122.354 133.802 122.254 133.919 122.119C134.277 121.582 134.431 120.934 134.352 120.293C134.484 118.966 134.715 117.65 135.043 116.357C135.509 114.273 136.209 111.696 137.141 108.627C138.073 105.559 139.239 102.182 140.637 98.4958C142.036 94.8102 143.643 91.1023 145.457 87.3721C148.097 86.9025 150.763 86.6017 153.441 86.4715C155.933 86.3548 158.108 86.3214 159.967 86.3715C162.137 86.4159 164.155 86.5271 166.019 86.705C166.338 86.8032 166.682 86.7795 166.985 86.6383C167.16 86.5329 167.291 86.3667 167.351 86.1713C167.368 86.0515 167.351 85.9295 167.305 85.818C167.258 85.7065 167.182 85.6097 167.085 85.5376C166.837 85.3261 166.555 85.1571 166.252 85.0373C165.874 84.8469 165.47 84.7122 165.054 84.637C161.85 83.866 158.59 83.3529 155.305 83.1027C152.792 82.9284 150.271 82.9061 147.755 83.036C147.888 82.7275 148.03 82.4273 148.188 82.1355C148.346 81.8436 148.488 81.5934 148.621 81.3683C148.754 81.1432 148.979 80.8596 149.154 80.6345L149.129 80.7012ZM148.463 75.3728C148.016 75.6105 147.592 75.8896 147.197 76.2067C146.814 76.4972 146.447 76.809 146.098 77.1406C145.704 77.5191 145.357 77.9448 145.066 78.4081C145.199 77.8728 145.4 77.3571 145.665 76.8738C145.933 76.4715 146.272 76.1212 146.664 75.8398C147.193 75.4848 147.831 75.331 148.463 75.4062V75.3728ZM134.885 105.892C135.373 104.341 135.84 102.876 136.284 101.498C136.733 100.122 137.183 98.7627 137.649 97.4285C138.115 96.0943 138.573 94.7685 139.014 93.4343C139.455 92.1001 139.921 90.6992 140.413 89.265C138.992 92.6004 137.704 95.9109 136.55 99.1963C136.067 100.619 135.579 102.098 135.085 103.632C134.602 105.158 134.144 106.693 133.728 108.227C133.312 109.761 132.945 111.271 132.662 112.763C132.379 114.256 132.163 115.64 132.03 116.933C132.254 115.907 132.521 114.798 132.862 113.597C133.079 112.58 133.37 111.404 133.695 110.07C134.019 108.736 134.444 107.36 134.885 105.901V105.892ZM255.11 115.899C255.441 117.469 255.373 119.098 254.91 120.635C254.542 121.773 254.006 122.849 253.32 123.829C252.583 124.79 251.653 125.586 250.589 126.163C252.152 125.564 253.498 124.506 254.452 123.128C255.101 122.124 255.509 120.982 255.643 119.793C255.768 118.479 255.585 117.154 255.11 115.924V115.899ZM251.056 98.1039C251.694 97.0356 252.491 96.0711 253.42 95.2438C254.181 94.5635 254.991 93.9393 255.842 93.3759C256.657 92.8556 257.528 92.4304 258.44 92.1085C257.854 92.2031 257.276 92.3368 256.708 92.5087C256.101 92.7186 255.511 92.9751 254.943 93.2759C254.253 93.6412 253.617 94.1015 253.054 94.6434C252.371 95.2817 251.736 95.9701 251.156 96.703C250.552 97.4695 250.048 98.3102 249.657 99.2046C249.257 100.121 248.977 101.085 248.825 102.073C248.676 103.102 248.709 104.149 248.925 105.167C248.944 105.454 249.012 105.736 249.124 106.001C249.249 106.412 249.405 106.813 249.59 107.201C249.815 107.693 250.123 108.294 250.523 109.002C250.317 108.616 250.139 108.214 249.99 107.802C249.69 106.995 249.488 106.155 249.391 105.3C249.263 104.189 249.297 103.066 249.491 101.965C249.739 100.578 250.266 99.2561 251.039 98.0789L251.056 98.1039Z"
                              fill="#d1d2d6"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M124.215 28.7015C124.215 28.7015 123.716 27.0755 123.041 24.6239C122.367 22.1724 121.526 18.8786 120.669 15.6015C120.261 13.9338 119.703 12.3411 119.387 10.7901C119.143 9.36023 119.015 7.91295 119.004 6.4624C118.962 5.00466 119.088 3.54695 119.378 2.11796L119.703 2.0596C120.477 3.30479 121.108 4.63348 121.585 6.02045C122.087 7.37508 122.474 8.76981 122.742 10.1898C123 11.7491 123.05 13.4585 123.241 15.1346C123.599 18.47 123.999 21.8555 124.19 24.3988C124.382 26.9421 124.531 28.6265 124.531 28.6265L124.215 28.7015Z"
                              fill="url(#paint0_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M125.496 35.706C124.931 35.1226 124.426 34.4828 123.99 33.7965C123.848 33.5463 123.524 33.3295 123.457 33.0877C123.449 32.9511 123.468 32.8142 123.514 32.6852C123.56 32.5563 123.631 32.4379 123.723 32.3372C124.01 32.0043 124.35 31.7224 124.731 31.5033L125.055 31.445C125.495 31.5276 125.918 31.6856 126.304 31.9119C126.425 31.9759 126.533 32.0634 126.62 32.1693C126.707 32.2752 126.772 32.3975 126.812 32.529C126.812 32.7708 126.612 33.0877 126.57 33.3629C126.398 34.1558 126.147 34.9296 125.821 35.6727L125.496 35.706Z"
                              fill="url(#paint1_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M109.68 21.9806C109.134 21.7053 108.623 21.3666 108.156 20.9717C108.007 20.8299 107.69 20.7465 107.624 20.5964C107.557 20.4463 107.682 20.2379 107.857 20.0544C108.14 19.7587 108.462 19.5033 108.814 19.2956L109.13 19.2039C109.54 19.1765 109.952 19.2045 110.354 19.2873C110.612 19.3456 110.812 19.4457 110.87 19.6041C110.929 19.7625 110.687 20.0127 110.654 20.2045C110.5 20.7853 110.276 21.3453 109.988 21.8722L109.68 21.9806ZM117.771 47.0882C117.771 47.0882 115.274 41.8015 113.135 36.4315C112.602 35.0806 111.894 33.7881 111.47 32.504C111.095 31.3214 110.844 30.103 110.72 28.8683C110.557 27.6173 110.557 26.3503 110.72 25.0993L111.028 25.0076C111.891 25.9449 112.632 26.987 113.234 28.1095C113.847 29.1915 114.351 30.331 114.741 31.5117C115.133 32.8125 115.316 34.2634 115.665 35.681C117.047 41.2929 118.113 47.0132 118.113 47.0132L117.771 47.0882Z"
                              fill="url(#paint2_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M105.983 44.695C105.983 44.695 103.095 40.3256 100.522 35.7811C99.8812 34.647 99.0737 33.588 98.5326 32.4956C98.0551 31.4847 97.6975 30.4212 97.467 29.3269C97.1885 28.2184 97.0707 27.0754 97.1174 25.9331L97.4088 25.7997C98.3289 26.5029 99.147 27.3308 99.8396 28.2596C100.535 29.1395 101.132 30.093 101.621 31.1031C102.121 32.2121 102.453 33.4963 102.895 34.7054C104.751 39.5668 106.308 44.5533 106.308 44.5533L105.983 44.695Z"
                              fill="url(#paint3_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M87.9256 35.9311C87.9256 35.9311 85.4282 33.4296 83.1889 30.6861C82.6395 30.0024 81.9319 29.4187 81.4657 28.7432C81.0625 28.1039 80.7672 27.4024 80.5916 26.6669C80.3763 25.8802 80.3058 25.0608 80.3835 24.2487L80.6582 24.082C81.4403 24.3653 82.167 24.7829 82.8059 25.3161C83.4099 25.7832 83.933 26.3466 84.3544 26.9838C84.7789 27.6759 85.012 28.5515 85.4033 29.3269C86.9433 32.4539 88.2004 35.7644 88.2004 35.7644L87.9256 35.9311ZM78.6853 21.6221C77.9668 21.3367 77.2803 20.9761 76.6374 20.5464C76.4293 20.3879 76.0547 20.3212 75.9132 20.1378C75.856 20.0225 75.8262 19.8955 75.8262 19.7667C75.8262 19.638 75.856 19.5109 75.9132 19.3956C76.0624 18.9939 76.2761 18.6193 76.5458 18.2866L76.8206 18.1115C77.2498 18.0187 77.6911 17.9961 78.1275 18.0448C78.2572 18.0507 78.3844 18.0825 78.5016 18.1384C78.6188 18.1942 78.7237 18.273 78.8101 18.37C78.9183 18.5785 78.8101 18.9287 78.8684 19.2039C78.9896 19.9506 79.026 20.7087 78.9766 21.4636L78.6853 21.6221Z"
                              fill="url(#paint4_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M93.7794 44.9952C92.8295 44.3721 91.9278 43.678 91.0822 42.9189C90.7908 42.6354 90.3413 42.4436 90.1166 42.1434C89.9336 41.8225 89.8578 41.4515 89.9001 41.0844C89.9432 40.5813 90.0906 40.0928 90.333 39.6501L90.6077 39.4834C91.1212 39.4608 91.6338 39.543 92.1145 39.7252C92.4718 39.8437 92.7832 40.0711 93.0052 40.3756C93.1354 40.762 93.2356 41.1578 93.3049 41.5597C93.6426 42.6243 93.8931 43.7147 94.0541 44.8201L93.7794 44.9952Z"
                              fill="url(#paint5_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M84.6123 51.3243C84.6123 51.3243 83.4635 50.2736 81.7903 48.6392C80.117 47.0049 77.9193 44.7951 75.7216 42.5937C74.6394 41.4764 73.4406 40.4591 72.475 39.375C71.6247 38.3521 70.8669 37.2554 70.2107 36.0979C69.5255 34.9482 68.9831 33.7188 68.5957 32.4373L68.8454 32.2288C70.0546 32.8457 71.1863 33.6043 72.2169 34.4886C73.254 35.3277 74.2151 36.2569 75.0889 37.2654C76.0046 38.3911 76.8038 39.7669 77.7195 40.9844C79.5259 43.486 81.374 45.9876 82.6643 47.9221C83.9547 49.8567 84.8454 51.1408 84.8454 51.1408L84.6123 51.3243Z"
                              fill="url(#paint6_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M96.033 70.6449C96.033 70.6449 92.2952 67.3595 88.7989 63.8656C87.9664 63.0317 86.9258 62.1978 86.1683 61.364C85.4908 60.5531 84.9097 59.666 84.4368 58.7206C83.923 57.7599 83.5528 56.7288 83.3379 55.6604L83.5876 55.4602C84.6157 55.8786 85.5797 56.44 86.4513 57.128C87.3148 57.748 88.1002 58.4704 88.7905 59.2793C89.5231 60.1716 90.0975 61.2806 90.8217 62.2812C93.6854 66.2754 96.2911 70.4781 96.2911 70.4781L96.033 70.6449Z"
                              fill="url(#paint7_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M78.0456 65.7251C77.2058 65.7141 76.3691 65.6191 75.5482 65.4416C75.2735 65.3665 74.8822 65.4416 74.6574 65.3165C74.4494 65.1368 74.3207 64.882 74.2995 64.6077C74.2469 64.1759 74.2723 63.7381 74.3744 63.3152L74.5575 63.0484C74.9272 62.7967 75.3388 62.6131 75.773 62.5064C75.909 62.4603 76.0529 62.4424 76.196 62.4539C76.3392 62.4654 76.4785 62.5059 76.6054 62.5731C76.8052 62.7315 76.8718 63.1068 77.0466 63.3319C77.51 63.9983 77.9033 64.7108 78.2204 65.4582L78.0372 65.7251H78.0456ZM72.0102 61.7225C72.0102 61.7225 70.4618 60.9304 68.1892 59.6796C65.9165 58.4288 62.9113 56.7444 59.8645 55.0433C58.3661 54.2094 56.7927 53.4423 55.4358 52.5417C54.2156 51.7144 53.0607 50.7944 51.9811 49.79C50.8832 48.7911 49.9003 47.6723 49.0508 46.4545L49.234 46.196C50.6887 46.5267 52.1037 47.0132 53.4546 47.6469C54.81 48.2427 56.1132 48.9512 57.3505 49.765C58.6741 50.6822 59.9644 51.8329 61.338 52.8669C64.0601 54.9766 66.8073 57.0362 68.8302 58.704C70.8531 60.3717 72.16 61.414 72.16 61.414L72.0102 61.7225Z"
                              fill="url(#paint8_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M56.9496 67.1677C56.3646 67.3629 55.758 67.4861 55.1432 67.5346C54.9351 67.5346 54.6521 67.693 54.4939 67.643C54.3357 67.5929 54.2774 67.3511 54.2608 67.0926C54.2417 66.6921 54.2781 66.2908 54.369 65.9002L54.5188 65.6167C54.7877 65.2981 55.0985 65.0176 55.4429 64.7828C55.6676 64.6494 55.8841 64.566 56.0339 64.6411C56.1838 64.7161 56.2087 65.058 56.3253 65.2164C56.6521 65.7377 56.9122 66.298 57.0995 66.8842L56.9496 67.1593V67.1677ZM81.0911 79.0669C81.0911 79.0669 75.4803 77.1073 70.0277 74.8559C68.6624 74.2805 67.2223 73.8803 65.9736 73.2882C64.8417 72.7259 63.7681 72.0528 62.7686 71.2786C61.7374 70.5242 60.8035 69.6447 59.9881 68.6603L60.138 68.3684C61.433 68.4199 62.7166 68.6328 63.959 69.0022C65.1697 69.3374 66.346 69.787 67.472 70.3447C68.6957 70.9701 69.8861 71.8623 71.1681 72.5961C76.2545 75.5563 81.1577 78.8001 81.1577 78.8001L81.0911 79.0669Z"
                              fill="url(#paint9_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M46.5435 150.754C46.2157 151.598 45.797 152.403 45.2948 153.155C45.1033 153.414 45.0285 153.839 44.812 153.989C44.6691 154.053 44.5145 154.085 44.3583 154.085C44.202 154.085 44.0475 154.053 43.9046 153.989C43.4078 153.84 42.9404 153.606 42.5227 153.297L42.3146 152.98C42.1807 152.489 42.1356 151.978 42.1814 151.471C42.1793 151.312 42.2126 151.155 42.2788 151.011C42.3451 150.867 42.4426 150.739 42.5643 150.637C42.8141 150.504 43.247 150.637 43.5633 150.537C44.476 150.382 45.4018 150.318 46.3271 150.345L46.5435 150.662V150.754ZM63.7756 139.605C63.7756 139.605 60.7454 142.599 57.4988 145.259C56.6663 145.926 55.992 146.76 55.1762 147.31C54.4064 147.802 53.5634 148.168 52.6788 148.394C51.7248 148.669 50.7315 148.782 49.7402 148.728L49.5321 148.402C49.8581 147.489 50.346 146.641 50.9722 145.901C51.5355 145.189 52.2069 144.57 52.9618 144.066C53.7943 143.549 54.8682 143.233 55.8089 142.774C59.5882 140.889 63.5924 139.338 63.5924 139.338L63.7756 139.605Z"
                              fill="url(#paint10_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M91.3044 145.151C91.3044 145.151 89.9309 146.268 87.8331 147.886C85.7353 149.503 82.9132 151.621 80.0912 153.723C78.6676 154.765 77.3274 155.932 75.9288 156.858C74.6485 157.673 73.307 158.386 71.9163 158.993C70.5255 159.632 69.0719 160.124 67.5792 160.46L67.4043 160.193C68.3009 158.973 69.3282 157.854 70.4679 156.858C71.5861 155.839 72.783 154.911 74.0474 154.081C75.4543 153.247 77.0776 152.464 78.6177 151.58C81.7228 149.87 84.7946 148.119 87.1671 146.902C89.5396 145.684 91.1047 144.842 91.1047 144.842L91.2795 145.109L91.3044 145.151ZM64.5907 162.428C63.9749 163.374 63.2845 164.269 62.5261 165.105C62.2431 165.397 62.0599 165.847 61.7602 166.064C61.4292 166.252 61.0492 166.336 60.6697 166.306C60.1584 166.275 59.6579 166.145 59.1962 165.922L59.0131 165.655C58.9812 165.154 59.0549 164.651 59.2296 164.179C59.3514 163.841 59.5754 163.55 59.8705 163.345C60.2634 163.206 60.6672 163.1 61.0776 163.029C62.1635 162.662 63.277 162.383 64.4075 162.195L64.5907 162.428Z"
                              fill="url(#paint11_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M86.2266 165.23C86.2266 165.23 85.1611 166.381 83.5294 168.048C81.8978 169.716 79.6834 171.909 77.4774 174.102C76.3619 175.186 75.3379 176.379 74.2557 177.338C73.2175 178.199 72.103 178.964 70.9259 179.622C69.7537 180.32 68.505 180.879 67.2047 181.29L66.9883 181.048C67.591 179.85 68.3374 178.73 69.211 177.713C70.0476 176.673 70.9784 175.713 71.9914 174.844C73.1236 173.927 74.4888 173.11 75.7375 172.184C78.2349 170.367 80.7906 168.507 82.7386 167.181C84.6865 165.855 85.9852 164.98 85.9852 164.98L86.2266 165.23Z"
                              fill="url(#paint12_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M99.0128 168.949C99.0201 169.758 98.942 170.566 98.7797 171.359C98.7131 171.634 98.7798 172.009 98.6715 172.193C98.5837 172.298 98.476 172.384 98.3545 172.447C98.233 172.51 98.1002 172.548 97.9639 172.559C97.518 172.621 97.065 172.606 96.6237 172.518L96.3489 172.343C96.0854 171.995 95.8879 171.602 95.7662 171.184C95.7112 171.05 95.6871 170.905 95.6958 170.761C95.7044 170.617 95.7456 170.476 95.8161 170.35C95.9743 170.15 96.3573 170.075 96.582 169.899C97.2604 169.431 97.9824 169.028 98.7381 168.699L99.0128 168.874V168.949ZM95.0503 174.928C95.0503 174.928 94.2761 176.454 93.0524 178.697C91.8287 180.94 90.1637 183.908 88.4988 186.877C87.6663 188.353 86.9254 189.912 86.068 191.255C85.2508 192.471 84.3347 193.617 83.3291 194.682C82.3309 195.783 81.2142 196.77 79.9993 197.625L79.7246 197.45C80.029 196.012 80.4954 194.614 81.1148 193.281C81.6955 191.943 82.3889 190.657 83.1876 189.437C84.095 188.128 85.2439 186.844 86.2761 185.484C88.3656 182.783 90.4384 180.056 92.0534 178.063C93.6684 176.07 94.7506 174.728 94.7506 174.728L95.0503 174.928Z"
                              fill="url(#paint13_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M101.018 189.479C101.232 190.042 101.372 190.63 101.434 191.23C101.434 191.43 101.617 191.705 101.568 191.855C101.518 192.005 101.268 192.072 101.01 192.097C100.598 192.124 100.184 192.099 99.7777 192.022L99.4863 191.88C99.1721 191.626 98.8922 191.331 98.6539 191.005C98.5842 190.928 98.5335 190.836 98.506 190.736C98.4785 190.636 98.475 190.531 98.4957 190.429C98.5873 190.296 98.9203 190.254 99.0784 190.129C99.6021 189.806 100.161 189.543 100.743 189.345L101.035 189.479H101.018ZM112.631 165.722C112.631 165.722 110.758 171.225 108.585 176.604C108.036 177.938 107.653 179.356 107.078 180.581C106.522 181.701 105.855 182.763 105.089 183.75C104.333 184.781 103.452 185.713 102.467 186.527L102.175 186.385C102.2 185.124 102.384 183.872 102.725 182.658C103.034 181.451 103.46 180.278 103.998 179.155C104.614 177.946 105.497 176.762 106.221 175.503C109.126 170.5 112.34 165.58 112.34 165.58L112.631 165.722Z"
                              fill="url(#paint14_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M119.576 175.286C119.576 175.286 118.636 180.415 117.378 185.418C117.054 186.677 116.912 187.994 116.546 189.153C116.181 190.209 115.701 191.222 115.114 192.172C114.537 193.173 113.825 194.091 113 194.899L112.692 194.799C112.508 193.668 112.486 192.517 112.625 191.38C112.733 190.271 112.962 189.178 113.308 188.119C113.716 186.969 114.39 185.835 114.889 184.634C116.921 179.839 119.268 175.153 119.268 175.153L119.576 175.286Z"
                              fill="url(#paint15_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M130.256 195.257C130.526 196.352 130.704 197.468 130.789 198.593C130.868 198.992 130.921 199.396 130.947 199.802C130.86 200.162 130.659 200.484 130.373 200.719C129.991 201.059 129.543 201.318 129.058 201.478L128.741 201.42C128.35 201.097 128.029 200.697 127.801 200.244C127.619 199.923 127.549 199.55 127.601 199.185C127.782 198.817 127.985 198.461 128.208 198.117C128.71 197.11 129.289 196.142 129.94 195.224L130.256 195.282V195.257ZM136.208 164.571C136.208 164.571 136.075 166.306 135.801 168.907C135.526 171.509 135.11 174.944 134.693 178.388C134.477 180.114 134.394 181.857 134.111 183.458C133.821 184.922 133.408 186.358 132.879 187.753C132.383 189.17 131.729 190.526 130.931 191.797L130.614 191.738C130.35 190.267 130.249 188.771 130.315 187.277C130.342 185.789 130.493 184.305 130.764 182.841C131.114 181.257 131.688 179.597 132.129 177.921C133.037 174.586 133.903 171.192 134.627 168.69C135.351 166.189 135.859 164.521 135.859 164.521L136.208 164.571Z"
                              fill="url(#paint16_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M144.309 185.543C144.309 185.543 144.417 187.077 144.5 189.387C144.584 191.697 144.65 194.757 144.725 197.834C144.725 199.368 144.908 200.911 144.85 202.345C144.763 203.66 144.554 204.963 144.226 206.239C143.921 207.544 143.46 208.806 142.852 210H142.527C142.066 208.747 141.759 207.442 141.612 206.114C141.437 204.807 141.384 203.487 141.453 202.17C141.57 200.736 141.911 199.226 142.111 197.7C142.552 194.657 142.944 191.613 143.335 189.362C143.726 187.11 143.984 185.559 143.984 185.559L144.309 185.543Z"
                              fill="url(#paint17_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M148.93 160.677C148.93 160.677 150.969 166.114 152.684 171.626C153.1 173.001 153.7 174.344 154.024 175.653C154.303 176.857 154.457 178.087 154.482 179.322C154.551 180.578 154.453 181.837 154.191 183.066L153.874 183.141C153.095 182.143 152.443 181.05 151.935 179.889C151.408 178.767 150.998 177.593 150.711 176.387C150.42 175.069 150.361 173.61 150.128 172.218C149.204 166.531 148.613 160.769 148.613 160.769L148.93 160.702V160.677ZM154.757 185.576C155.354 186.134 155.892 186.751 156.364 187.419C156.513 187.652 156.855 187.853 156.93 188.094C156.946 188.23 156.934 188.367 156.895 188.497C156.857 188.628 156.792 188.749 156.705 188.853C156.435 189.205 156.109 189.509 155.739 189.754L155.423 189.821C154.978 189.762 154.547 189.626 154.149 189.42C154.024 189.366 153.911 189.286 153.818 189.185C153.724 189.085 153.653 188.966 153.608 188.837C153.608 188.586 153.783 188.27 153.808 188.003C153.935 187.201 154.147 186.416 154.441 185.66L154.757 185.576Z"
                              fill="url(#paint18_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M173.468 192.856C174.028 193.104 174.556 193.417 175.041 193.79C175.2 193.923 175.516 193.99 175.599 194.132C175.682 194.273 175.549 194.49 175.391 194.69C175.127 195.01 174.819 195.291 174.475 195.524L174.159 195.632C173.753 195.686 173.341 195.686 172.935 195.632C172.677 195.582 172.461 195.499 172.403 195.341C172.344 195.182 172.561 194.924 172.594 194.724C172.716 194.146 172.909 193.586 173.168 193.056L173.468 192.947V192.856ZM164.111 168.165C164.111 168.165 166.85 173.318 169.281 178.588C169.88 179.906 170.654 181.165 171.154 182.424C171.593 183.586 171.911 184.791 172.103 186.018C172.329 187.258 172.396 188.522 172.303 189.779L172.003 189.887C171.093 188.998 170.301 187.994 169.647 186.902C168.983 185.848 168.425 184.731 167.982 183.566C167.524 182.291 167.275 180.84 166.85 179.464C165.185 173.919 163.828 168.273 163.828 168.273L164.111 168.165Z"
                              fill="url(#paint19_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M176.768 161.319C176.768 161.319 179.881 165.563 182.703 169.958C183.403 171.059 184.26 172.084 184.859 173.143C185.388 174.125 185.799 175.167 186.083 176.245C186.419 177.343 186.596 178.483 186.608 179.631L186.325 179.781C185.371 179.116 184.511 178.325 183.769 177.429C183.026 176.588 182.38 175.665 181.846 174.678C181.288 173.594 180.922 172.326 180.389 171.134C178.283 166.381 176.477 161.469 176.477 161.469L176.768 161.319Z"
                              fill="url(#paint20_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M194.474 177.913C194.474 177.913 197.113 180.314 199.469 182.916C200.06 183.567 200.792 184.125 201.284 184.767C201.722 185.386 202.054 186.074 202.266 186.802C202.521 187.576 202.634 188.389 202.599 189.203L202.333 189.387C201.536 189.147 200.789 188.765 200.126 188.261C199.489 187.827 198.927 187.292 198.462 186.677C198.004 186.001 197.729 185.143 197.296 184.375C195.631 181.323 194.166 178.088 194.166 178.088L194.432 177.913H194.474ZM204.464 191.755C205.195 192.005 205.898 192.332 206.561 192.731C206.786 192.872 207.161 192.922 207.311 193.098C207.371 193.211 207.407 193.335 207.416 193.463C207.424 193.591 207.406 193.72 207.361 193.84C207.235 194.25 207.041 194.637 206.786 194.982L206.52 195.166C206.097 195.282 205.658 195.327 205.221 195.299C205.091 195.302 204.961 195.277 204.841 195.227C204.721 195.176 204.612 195.102 204.522 195.007C204.405 194.799 204.522 194.448 204.422 194.173C204.265 193.431 204.189 192.673 204.197 191.914L204.464 191.755Z"
                              fill="url(#paint21_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M189.661 154.215C189.661 154.215 190.86 155.215 192.617 156.766C194.373 158.317 196.679 160.41 198.985 162.503C200.125 163.571 201.374 164.53 202.39 165.572C203.295 166.544 204.106 167.599 204.812 168.724C205.559 169.84 206.166 171.044 206.619 172.309L206.377 172.518C205.134 171.97 203.961 171.275 202.881 170.45C201.803 169.659 200.798 168.775 199.876 167.807C198.902 166.731 198.028 165.43 197.054 164.221C195.122 161.794 193.158 159.401 191.759 157.55C190.361 155.699 189.42 154.44 189.42 154.44L189.661 154.215Z"
                              fill="url(#paint22_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M202.797 147.644C203.634 147.619 204.47 147.677 205.295 147.819C205.596 147.812 205.898 147.838 206.194 147.894C206.305 147.975 206.397 148.078 206.464 148.197C206.532 148.317 206.573 148.449 206.585 148.586C206.664 149.016 206.664 149.457 206.585 149.887L206.41 150.154C206.054 150.421 205.654 150.624 205.228 150.754C205.096 150.81 204.955 150.838 204.812 150.838C204.669 150.838 204.528 150.81 204.396 150.754C204.188 150.604 204.096 150.229 203.913 150.012C203.413 149.371 202.983 148.677 202.631 147.944L202.797 147.669V147.644ZM209.033 151.354C209.033 151.354 210.614 152.072 212.953 153.206C215.293 154.34 218.373 155.891 221.47 157.442C223.001 158.226 224.608 158.893 226.015 159.693C227.276 160.46 228.476 161.324 229.603 162.278C230.751 163.205 231.79 164.259 232.7 165.422L232.533 165.697C231.061 165.434 229.621 165.014 228.238 164.446C226.858 163.911 225.522 163.27 224.242 162.528C222.868 161.694 221.528 160.585 220.079 159.618C217.249 157.642 214.402 155.716 212.321 154.173C210.24 152.63 208.841 151.671 208.841 151.671L209.033 151.354Z"
                              fill="url(#paint23_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M208.451 133.26C208.451 133.26 214.154 134.927 219.723 136.929C221.113 137.429 222.57 137.762 223.885 138.305C225.053 138.81 226.169 139.43 227.215 140.156C228.285 140.864 229.261 141.705 230.12 142.657L229.996 142.957C228.7 142.961 227.409 142.81 226.15 142.507C224.909 142.238 223.697 141.853 222.528 141.356C221.271 140.789 220.031 139.956 218.724 139.28C213.488 136.57 208.385 133.568 208.385 133.568L208.518 133.285L208.451 133.26ZM233.134 143.975C233.708 143.755 234.306 143.604 234.915 143.524C235.115 143.524 235.398 143.333 235.556 143.383C235.715 143.433 235.79 143.658 235.823 143.916C235.857 144.317 235.837 144.721 235.765 145.117L235.631 145.409C235.395 145.725 235.114 146.006 234.799 146.243C234.722 146.315 234.629 146.368 234.528 146.396C234.426 146.425 234.32 146.43 234.216 146.41C234.066 146.326 234.016 146.009 233.892 145.851C233.553 145.357 233.274 144.826 233.059 144.267L233.134 143.975Z"
                              fill="url(#paint24_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M210.059 67.6096C210.059 67.6096 215.128 63.8906 220.389 60.5051C221.713 59.6712 222.887 58.6372 224.202 57.9201C225.395 57.2869 226.65 56.781 227.948 56.4108C229.279 55.9949 230.659 55.7593 232.052 55.7104L232.235 56.0356C231.456 57.166 230.523 58.182 229.463 59.0542C228.442 59.9421 227.325 60.714 226.134 61.3556C224.835 62.031 223.295 62.498 221.871 63.1484C216.136 65.7167 210.217 67.9431 210.217 67.9431L210.059 67.6096Z"
                              fill="url(#paint25_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M219.764 41.3095C220.003 40.5874 220.322 39.8939 220.713 39.2415C220.863 39.0247 220.896 38.6578 221.071 38.4994C221.188 38.4391 221.316 38.4029 221.448 38.3928C221.579 38.3828 221.711 38.3992 221.837 38.441C222.257 38.5501 222.656 38.7276 223.019 38.9664L223.218 39.2249C223.348 39.6341 223.405 40.0633 223.385 40.4923C223.388 40.6198 223.365 40.7466 223.316 40.8644C223.267 40.9822 223.194 41.0884 223.102 41.1761C222.894 41.2929 222.527 41.1761 222.269 41.2929C221.512 41.4637 220.739 41.5559 219.963 41.568L219.764 41.3095Z"
                              fill="#E64290"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M205.803 51.3576C205.803 51.3576 208.217 48.7226 210.798 46.3545C211.455 45.7707 211.996 45.0453 212.654 44.545C213.279 44.0988 213.974 43.7606 214.71 43.5443C215.513 43.2694 216.359 43.1395 217.208 43.1608L217.399 43.4109C217.168 44.1974 216.796 44.935 216.3 45.5873C215.86 46.211 215.324 46.76 214.71 47.2133C214.028 47.6803 213.145 47.9722 212.379 48.4058C209.291 50.1402 206.003 51.6078 206.003 51.6078L205.803 51.3576Z"
                              fill="url(#paint26_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M189.787 49.2813C189.787 49.2813 190.786 48.0805 192.334 46.3294C193.883 44.5783 195.981 42.2935 198.07 39.992C199.136 38.858 200.085 37.6155 201.125 36.5982C202.114 35.6933 203.181 34.8785 204.314 34.1633C205.447 33.4067 206.665 32.7855 207.943 32.3122L208.168 32.5456C207.63 33.7764 206.942 34.9359 206.12 35.9978C205.334 37.0746 204.453 38.0792 203.49 38.9997C202.407 39.967 201.092 40.8509 199.885 41.8349C197.437 43.7694 195.023 45.7457 193.142 47.1549C191.261 48.5641 190.012 49.4981 190.012 49.4981L189.787 49.2813Z"
                              fill="url(#paint27_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M178.209 37.824C178.209 37.824 178.908 36.273 180.015 33.9716C181.123 31.6701 182.638 28.6265 184.178 25.5746C184.952 24.0569 185.593 22.4726 186.384 21.0884C187.137 19.8362 187.992 18.6481 188.939 17.5361C189.875 16.4024 190.933 15.376 192.095 14.4759L192.378 14.6259C192.143 16.0552 191.749 17.4537 191.204 18.7953C190.688 20.1589 190.061 21.478 189.331 22.7394C188.498 24.0903 187.399 25.4328 186.442 26.842C184.486 29.6355 182.579 32.4623 181.048 34.5303C179.516 36.5982 178.55 37.9825 178.55 37.9825L178.209 37.824Z"
                              fill="url(#paint28_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M174.562 44.0029C174.513 43.1953 174.549 42.3847 174.67 41.5847C174.72 41.3095 174.62 40.9426 174.729 40.7509C174.813 40.6348 174.92 40.5372 175.043 40.464C175.166 40.3909 175.303 40.3438 175.445 40.3256C175.885 40.2482 176.336 40.2482 176.776 40.3256L177.06 40.484C177.343 40.8228 177.563 41.2097 177.709 41.6264C177.763 41.75 177.791 41.8834 177.791 42.0183C177.791 42.1532 177.763 42.2867 177.709 42.4102C177.559 42.6187 177.184 42.7188 176.96 42.9022C176.307 43.4029 175.607 43.8385 174.87 44.203L174.562 44.0029Z"
                              fill="#866EAB"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M171.506 23.59C171.266 23.0385 171.098 22.4583 171.006 21.8639C171.006 21.6721 170.798 21.4053 170.839 21.2468C170.881 21.0884 171.123 21.0133 171.381 20.9716C171.789 20.9159 172.204 20.9159 172.613 20.9716L172.921 21.0967C173.252 21.3327 173.549 21.6133 173.803 21.9306C173.953 22.1307 174.053 22.3392 173.986 22.4893C173.92 22.6394 173.578 22.6977 173.42 22.8228C172.914 23.16 172.373 23.4396 171.805 23.6567L171.506 23.59Z"
                              fill="#CDD528"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M161.125 47.8887C161.125 47.8887 162.715 42.2935 164.613 36.8234C165.088 35.4642 165.446 34.0216 165.92 32.7708C166.413 31.625 167.021 30.5326 167.735 29.5104C168.443 28.4451 169.282 27.473 170.232 26.6169L170.532 26.742C170.572 28.0033 170.449 29.2646 170.166 30.4944C169.924 31.7127 169.559 32.903 169.075 34.0466C168.526 35.2891 167.701 36.5482 167.044 37.799C164.397 42.9689 161.441 48.0055 161.441 48.0055L161.125 47.8887Z"
                              fill="url(#paint29_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M153.699 38.6579C153.699 38.6579 154.382 33.4879 155.364 28.393C155.614 27.1256 155.697 25.808 156.005 24.6323C156.309 23.558 156.736 22.5226 157.279 21.547C157.802 20.5177 158.466 19.5663 159.252 18.7202L159.568 18.8036C159.804 19.924 159.883 21.0719 159.801 22.2141C159.762 23.3425 159.589 24.4623 159.285 25.5496C158.944 26.7086 158.328 27.8844 157.886 29.1102C156.105 33.9882 153.999 38.7829 153.999 38.7829L153.699 38.6579Z"
                              fill="url(#paint30_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M148.021 3.90248C147.666 3.23511 147.387 2.53033 147.188 1.80115C147.13 1.54265 146.888 1.25914 146.897 1.03399C146.931 0.90868 146.991 0.792271 147.075 0.692896C147.158 0.593521 147.262 0.513572 147.379 0.458627C147.757 0.241055 148.168 0.0861739 148.595 0L148.92 0.0416957C149.311 0.234436 149.666 0.494036 149.968 0.808848C150.065 0.892674 150.143 0.996395 150.196 1.11295C150.249 1.2295 150.277 1.35614 150.277 1.48427C150.218 1.70942 149.91 1.91789 149.777 2.14303C149.373 2.79383 148.901 3.40031 148.37 3.95251L148.021 3.90248Z"
                              fill="#3BB9C5"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M145.65 20.6548C145.65 20.6548 145.4 17.1359 145.483 13.667C145.483 12.7915 145.358 11.9076 145.483 11.0987C145.597 10.3534 145.833 9.6321 146.182 8.96403C146.542 8.22241 147.029 7.55047 147.623 6.97943L147.939 7.02947C148.353 7.7341 148.635 8.50833 148.771 9.31425C148.926 10.0519 148.954 10.8103 148.855 11.5573C148.722 12.3495 148.322 13.175 148.114 14.0089C147.223 17.3444 146.008 20.6798 146.008 20.6798L145.65 20.6548Z"
                              fill="url(#paint31_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M144.149 31.2531C143.832 30.1809 143.601 29.0851 143.458 27.9761C143.351 27.5848 143.273 27.1863 143.225 26.7836C143.299 26.4189 143.485 26.0865 143.757 25.833C144.121 25.4715 144.554 25.1878 145.031 24.9992L145.356 25.0409C145.761 25.3453 146.1 25.7281 146.355 26.1666C146.551 26.4829 146.639 26.8548 146.604 27.2256C146.445 27.6027 146.259 27.9678 146.047 28.318C145.605 29.3518 145.078 30.3475 144.473 31.2948L144.149 31.2531Z"
                              fill="url(#paint32_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M134.037 30.1942C134.037 30.1942 133.763 28.6766 133.413 26.4001C133.063 24.1237 132.647 21.0717 132.231 18.0198C132.031 16.5022 131.698 14.9845 131.59 13.5586C131.525 12.2413 131.584 10.9208 131.765 9.61446C131.914 8.28833 132.228 6.98592 132.697 5.737H133.022C133.629 6.932 134.085 8.19811 134.379 9.50605C134.702 10.7824 134.905 12.0861 134.986 13.4002C135.036 14.8428 134.862 16.3771 134.837 17.9114C134.753 20.9883 134.712 24.057 134.579 26.3584C134.445 28.6599 134.37 30.2025 134.37 30.2025L134.037 30.1942Z"
                              fill="url(#paint33_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M135.293 52.9753C135.293 52.9753 134.286 48.1556 133.628 43.3275C133.47 42.1184 133.129 40.926 133.046 39.7919C132.993 38.7447 133.066 37.695 133.262 36.665C133.439 35.5916 133.769 34.5496 134.244 33.5713H134.569C135.147 34.4945 135.588 35.4974 135.876 36.5482C136.19 37.5497 136.38 38.5858 136.442 39.6335C136.442 40.7759 136.267 42.0017 136.242 43.2191C136.109 48.0889 135.635 52.9753 135.635 52.9753H135.293Z"
                              fill="url(#paint34_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M117.879 62.006L119.543 67.0926H124.888C125.066 67.0894 125.241 67.1437 125.386 67.2475C125.532 67.3513 125.64 67.4991 125.695 67.6691C125.75 67.8392 125.749 68.0224 125.692 68.1918C125.635 68.3613 125.525 68.5079 125.379 68.6102L121.05 71.7455L122.715 76.8321C122.773 77.0399 122.748 77.262 122.645 77.4517C122.543 77.6413 122.371 77.7839 122.166 77.8494C122.026 77.9001 121.874 77.9121 121.728 77.8842C121.581 77.8563 121.445 77.7896 121.333 77.691L117.071 74.5723L112.751 77.7243C112.662 77.7891 112.562 77.8358 112.455 77.8617C112.349 77.8876 112.238 77.8921 112.13 77.8751C112.021 77.858 111.918 77.8197 111.824 77.7624C111.731 77.705 111.649 77.6298 111.585 77.5409C111.508 77.434 111.457 77.3102 111.437 77.1797C111.416 77.0492 111.427 76.9158 111.469 76.7904L113.134 71.7038L108.805 68.5685C108.653 68.464 108.54 68.3118 108.483 68.1359C108.427 67.9601 108.43 67.7704 108.493 67.5968C108.556 67.4231 108.675 67.2753 108.831 67.1765C108.986 67.0777 109.17 67.0335 109.354 67.0509H114.674L116.338 61.9643C116.389 61.7911 116.494 61.6389 116.638 61.5307C116.782 61.4224 116.958 61.3639 117.138 61.3639C117.318 61.3639 117.493 61.4224 117.637 61.5307C117.781 61.6389 117.886 61.7911 117.937 61.9643L117.879 62.006ZM124.538 67.843L119.885 67.9097C121.55 67.9097 122.607 68.335 124.538 67.843ZM116.613 62.6814L115.24 67.1343C115.739 65.5249 116.488 64.6744 116.613 62.6814ZM112.293 76.1817C113.35 74.5139 113.233 73.3632 113.766 71.7622L112.293 76.1817ZM118.12 68.1349L117.088 64.9329L116.055 68.1349C115.999 68.309 115.887 68.4597 115.736 68.5637C115.586 68.6677 115.406 68.7192 115.223 68.7102H111.893L114.615 70.6782C114.759 70.7826 114.866 70.9305 114.92 71.1C114.974 71.2695 114.972 71.4519 114.915 71.6204L113.841 74.8391L116.538 72.8629C116.682 72.7546 116.858 72.6961 117.038 72.6961C117.218 72.6961 117.393 72.7546 117.537 72.8629L120.251 74.8475L119.219 71.6705C119.155 71.4995 119.149 71.3123 119.202 71.1376C119.254 70.9629 119.363 70.8103 119.51 70.7032L122.232 68.7353H118.902C118.726 68.7346 118.554 68.6776 118.412 68.5727C118.27 68.4678 118.165 68.3203 118.112 68.1516L118.12 68.1349Z"
                              fill="url(#paint35_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M217.9 80.1592C217.9 80.1592 219.149 79.4587 221.055 78.4915C222.962 77.5242 225.517 76.2234 228.073 74.9392C229.355 74.3138 230.571 73.5633 231.811 73.038C232.956 72.5818 234.14 72.2301 235.349 71.9873C236.57 71.7142 237.819 71.5855 239.07 71.6038L239.195 71.8789C238.324 72.7568 237.352 73.5267 236.298 74.172C235.268 74.8247 234.181 75.3828 233.051 75.8398C231.803 76.3151 230.379 76.6736 229.022 77.0822C226.283 77.9161 223.561 78.8417 221.48 79.4337C219.399 80.0258 218.017 80.426 218.017 80.426L217.9 80.1592Z"
                              fill="url(#paint36_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M151.244 53.9593L152.585 58.1286H156.947C157.036 58.1286 157.125 58.1462 157.208 58.1806C157.291 58.215 157.366 58.2653 157.429 58.3288C157.493 58.3923 157.543 58.4677 157.577 58.5507C157.612 58.6336 157.629 58.7225 157.629 58.8123C157.631 58.9214 157.606 59.0292 157.556 59.1265C157.507 59.2238 157.435 59.3077 157.346 59.371L153.8 61.906L155.157 66.0753C155.199 66.2171 155.195 66.3687 155.144 66.5079C155.094 66.647 155.001 66.7664 154.878 66.8485C154.755 66.9306 154.609 66.971 154.462 66.964C154.314 66.9569 154.173 66.9026 154.058 66.8091L150.57 64.2574L147.049 66.8341C146.976 66.8875 146.894 66.926 146.806 66.9474C146.719 66.9688 146.628 66.9726 146.539 66.9587C146.45 66.9447 146.365 66.9133 146.288 66.8661C146.211 66.819 146.144 66.7571 146.091 66.684C146.029 66.5969 145.988 66.4968 145.971 66.3912C145.954 66.2857 145.961 66.1776 145.991 66.0753L147.348 61.906L143.81 59.346C143.665 59.2378 143.568 59.0766 143.54 58.8972C143.512 58.7179 143.555 58.5346 143.661 58.3871C143.729 58.2914 143.821 58.2151 143.927 58.1654C144.034 58.1158 144.151 58.0945 144.268 58.1035H148.58L149.921 53.9342C149.978 53.7687 150.096 53.6311 150.25 53.5491C150.405 53.4671 150.584 53.4468 150.753 53.4923C150.857 53.5275 150.951 53.5866 151.027 53.6648C151.104 53.7431 151.161 53.8382 151.194 53.9426L151.244 53.9593ZM156.697 58.7039L152.901 58.7623C154.275 58.7623 155.115 59.1125 156.697 58.7039ZM150.22 54.5013L149.105 58.1452C149.504 56.8277 150.121 56.1356 150.22 54.5013ZM146.674 65.6C147.507 64.2158 147.448 63.2985 147.89 61.9977L146.674 65.6ZM151.436 59.0291L150.603 56.4108L149.771 59.0291C149.727 59.167 149.64 59.2873 149.524 59.3727C149.407 59.4581 149.266 59.5043 149.122 59.5044H146.374L148.597 61.1138C148.715 61.1986 148.802 61.3186 148.847 61.4565C148.892 61.5944 148.892 61.7431 148.847 61.8809L147.973 64.5076L150.179 62.8899C150.299 62.8005 150.445 62.7523 150.595 62.7523C150.745 62.7523 150.891 62.8005 151.011 62.8899L153.226 64.5576L152.393 61.9643C152.344 61.8237 152.341 61.6711 152.384 61.5287C152.428 61.3862 152.515 61.2613 152.635 61.1722L154.857 59.5628H152.135C151.972 59.5699 151.812 59.5183 151.684 59.4173C151.555 59.3164 151.467 59.1728 151.436 59.0125V59.0291Z"
                              fill="url(#paint37_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M184.793 60.2049L185.742 63.1401H188.822C188.885 63.1401 188.948 63.1526 189.007 63.1769C189.065 63.2012 189.118 63.2368 189.163 63.2817C189.208 63.3266 189.244 63.38 189.268 63.4386C189.292 63.4973 189.305 63.5602 189.305 63.6237C189.307 63.7006 189.29 63.7768 189.255 63.8452C189.22 63.9137 189.168 63.9723 189.105 64.0156L186.607 65.8335L187.565 68.7687C187.586 68.8294 187.594 68.8937 187.59 68.9578C187.586 69.0218 187.569 69.0844 187.54 69.1419C187.511 69.1993 187.471 69.2505 187.423 69.2923C187.374 69.3342 187.318 69.3659 187.257 69.3857C187.177 69.4089 187.094 69.4127 187.013 69.3968C186.931 69.3808 186.855 69.3456 186.791 69.294L184.293 67.4928L181.796 69.3107C181.712 69.3678 181.613 69.3978 181.512 69.3964C181.411 69.3951 181.313 69.3626 181.231 69.3033C181.149 69.2441 181.087 69.1609 181.054 69.0652C181.021 68.9695 181.019 68.866 181.047 68.7687L182.004 65.8335L179.507 64.0156C179.455 63.9784 179.412 63.9314 179.378 63.8773C179.345 63.8232 179.323 63.7631 179.313 63.7004C179.303 63.6377 179.305 63.5736 179.32 63.5118C179.334 63.45 179.361 63.3917 179.398 63.3402C179.447 63.274 179.511 63.221 179.585 63.1861C179.659 63.1511 179.741 63.1353 179.823 63.1401H182.936L183.885 60.1965C183.915 60.0973 183.977 60.0103 184.06 59.9485C184.143 59.8867 184.244 59.8534 184.347 59.8534C184.451 59.8534 184.552 59.8867 184.635 59.9485C184.718 60.0103 184.779 60.0973 184.809 60.1965L184.793 60.2049ZM188.647 63.5403H185.958C186.85 63.6729 187.756 63.6729 188.647 63.5403ZM184.068 60.5551L183.236 63.1317C183.639 62.3189 183.92 61.4505 184.068 60.5551ZM181.571 68.4101C181.972 67.605 182.252 66.7453 182.403 65.8585L181.571 68.4101ZM184.901 63.7571L184.302 61.906L183.71 63.7571C183.676 63.8539 183.613 63.9378 183.53 63.9975C183.446 64.0572 183.347 64.0897 183.244 64.0907H181.305L182.878 65.2247C182.961 65.2849 183.022 65.3698 183.054 65.4672C183.085 65.5646 183.085 65.6695 183.053 65.7667L182.453 67.6179L184.01 66.4755C184.093 66.4126 184.194 66.3786 184.297 66.3786C184.401 66.3786 184.502 66.4126 184.585 66.4755L186.158 67.6179L185.559 65.7918C185.522 65.6911 185.519 65.5812 185.551 65.4789C185.583 65.3765 185.647 65.2873 185.733 65.2247L187.298 64.0907H185.434C185.326 64.1027 185.217 64.078 185.125 64.0205C185.033 63.963 184.963 63.8761 184.926 63.7738L184.901 63.7571Z"
                              fill="url(#paint38_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M177.799 148.419L176.134 143.333H170.789C170.569 143.333 170.357 143.245 170.201 143.088C170.045 142.932 169.957 142.72 169.957 142.499C169.957 142.367 169.987 142.237 170.047 142.12C170.106 142.003 170.192 141.901 170.298 141.823L174.627 138.688L172.962 133.602C172.927 133.497 172.913 133.387 172.921 133.276C172.929 133.166 172.958 133.059 173.008 132.961C173.058 132.862 173.126 132.774 173.21 132.703C173.293 132.631 173.39 132.576 173.495 132.543C173.635 132.492 173.786 132.48 173.933 132.508C174.079 132.536 174.216 132.602 174.327 132.701L178.59 135.82L182.91 132.668C182.999 132.603 183.099 132.556 183.206 132.53C183.312 132.504 183.423 132.5 183.531 132.517C183.639 132.534 183.743 132.572 183.837 132.63C183.93 132.687 184.011 132.762 184.076 132.851C184.153 132.958 184.204 133.082 184.224 133.212C184.244 133.343 184.234 133.476 184.192 133.602L182.527 138.688L186.856 141.823C187.008 141.928 187.121 142.08 187.178 142.256C187.234 142.432 187.23 142.621 187.168 142.795C187.105 142.969 186.986 143.117 186.83 143.215C186.674 143.314 186.49 143.358 186.307 143.341H181.021L179.356 148.436C179.305 148.609 179.2 148.761 179.056 148.87C178.912 148.978 178.737 149.036 178.556 149.036C178.376 149.036 178.201 148.978 178.057 148.87C177.913 148.761 177.808 148.609 177.757 148.436L177.799 148.419ZM171.139 142.582L175.793 142.516C174.128 142.516 173.07 142.09 171.139 142.582ZM179.064 147.752L180.438 143.299C179.938 144.909 179.197 145.751 179.064 147.752ZM183.401 134.16C182.344 135.828 182.461 136.979 181.928 138.58L183.401 134.16ZM177.574 142.215L178.606 145.417L179.639 142.215C179.694 142.04 179.805 141.887 179.956 141.781C180.106 141.676 180.287 141.623 180.471 141.632H183.801L181.079 139.655C180.937 139.551 180.831 139.405 180.777 139.237C180.724 139.069 180.724 138.889 180.779 138.721L181.82 135.519L179.122 137.496C178.978 137.604 178.803 137.662 178.623 137.662C178.443 137.662 178.268 137.604 178.124 137.496L175.41 135.511L176.442 138.68C176.506 138.851 176.512 139.038 176.459 139.213C176.407 139.387 176.298 139.54 176.151 139.647L173.428 141.623H176.758C176.942 141.622 177.121 141.683 177.267 141.794C177.413 141.906 177.518 142.063 177.566 142.24L177.574 142.215Z"
                              fill="url(#paint39_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M142.612 151.846L141.272 147.677H136.909C136.764 147.678 136.622 147.633 136.504 147.548C136.386 147.462 136.298 147.342 136.254 147.203C136.209 147.064 136.21 146.915 136.256 146.777C136.302 146.638 136.391 146.519 136.51 146.435L140.048 143.875L138.691 139.705C138.643 139.536 138.663 139.355 138.745 139.2C138.827 139.045 138.966 138.927 139.132 138.871C139.243 138.832 139.362 138.823 139.478 138.845C139.593 138.867 139.701 138.919 139.79 138.997L143.269 141.548L146.799 138.972C146.872 138.918 146.954 138.88 147.042 138.858C147.129 138.837 147.22 138.833 147.309 138.847C147.398 138.861 147.483 138.892 147.56 138.939C147.637 138.987 147.704 139.049 147.756 139.122C147.819 139.208 147.861 139.309 147.876 139.415C147.892 139.521 147.883 139.629 147.848 139.73L146.491 143.9L150.021 146.46C150.168 146.567 150.267 146.727 150.296 146.907C150.326 147.086 150.284 147.27 150.179 147.419C150.109 147.513 150.017 147.588 149.911 147.638C149.805 147.688 149.688 147.71 149.571 147.702H145.259L143.919 151.871C143.862 152.037 143.744 152.176 143.59 152.259C143.436 152.342 143.256 152.365 143.086 152.322C142.983 152.285 142.89 152.226 142.814 152.148C142.737 152.07 142.679 151.975 142.645 151.871L142.612 151.846ZM137.151 147.102L140.947 147.043C139.573 147.043 138.724 146.693 137.151 147.102ZM143.619 151.321L144.743 147.677C144.335 148.995 143.727 149.687 143.619 151.321ZM147.14 140.214C146.308 141.607 146.375 142.524 145.933 143.825L147.14 140.214ZM142.387 146.793L143.219 149.411L144.052 146.793C144.095 146.655 144.181 146.534 144.298 146.448C144.415 146.362 144.556 146.317 144.701 146.318H147.44L145.217 144.708C145.102 144.622 145.016 144.502 144.971 144.365C144.926 144.227 144.925 144.079 144.968 143.941L145.8 141.323L143.594 142.941C143.475 143.033 143.328 143.083 143.178 143.083C143.027 143.083 142.881 143.033 142.762 142.941L140.547 141.323L141.38 143.916C141.433 144.056 141.438 144.21 141.394 144.354C141.351 144.497 141.261 144.622 141.138 144.708L138.916 146.318H141.629C141.797 146.286 141.97 146.316 142.116 146.404C142.262 146.492 142.37 146.63 142.42 146.793H142.387Z"
                              fill="url(#paint40_linear_148_1506)"/>
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M108.954 149.436L108.013 146.493H104.924C104.822 146.493 104.723 146.461 104.64 146.4C104.558 146.34 104.496 146.256 104.465 146.158C104.433 146.061 104.434 145.956 104.465 145.859C104.497 145.762 104.559 145.677 104.641 145.617L107.139 143.808L106.181 140.864C106.161 140.804 106.152 140.74 106.157 140.677C106.161 140.613 106.178 140.551 106.207 140.495C106.236 140.438 106.275 140.388 106.324 140.346C106.372 140.305 106.429 140.274 106.489 140.256C106.568 140.229 106.653 140.223 106.735 140.239C106.817 140.255 106.893 140.292 106.956 140.347L109.453 142.148L111.95 140.322C112.055 140.247 112.186 140.216 112.314 140.236C112.442 140.256 112.556 140.326 112.633 140.431C112.676 140.493 112.704 140.565 112.715 140.64C112.727 140.715 112.722 140.792 112.7 140.864L111.742 143.808L114.24 145.617C114.291 145.655 114.335 145.702 114.368 145.756C114.401 145.81 114.423 145.87 114.433 145.933C114.443 145.995 114.441 146.059 114.426 146.121C114.412 146.183 114.385 146.241 114.348 146.293C114.301 146.361 114.237 146.415 114.163 146.45C114.088 146.485 114.005 146.5 113.923 146.493H110.868L109.919 149.436C109.9 149.497 109.869 149.554 109.829 149.603C109.788 149.652 109.738 149.692 109.681 149.722C109.625 149.751 109.563 149.769 109.499 149.774C109.436 149.78 109.372 149.773 109.311 149.753C109.238 149.729 109.171 149.687 109.116 149.632C109.061 149.577 109.02 149.51 108.995 149.436H108.954ZM105.099 146.101L107.788 146.059C106.895 145.941 105.989 145.955 105.099 146.101ZM109.678 149.086L110.51 146.518C110.107 147.328 109.826 148.193 109.678 149.086ZM112.175 141.24C111.776 142.045 111.496 142.905 111.343 143.791L112.175 141.24ZM108.845 145.893L109.445 147.735L110.036 145.893C110.067 145.794 110.13 145.707 110.213 145.646C110.297 145.584 110.398 145.551 110.502 145.551H112.442L110.868 144.408C110.784 144.349 110.722 144.264 110.691 144.166C110.659 144.069 110.66 143.963 110.693 143.866L111.293 142.015L109.678 143.141C109.594 143.201 109.494 143.234 109.391 143.234C109.287 143.234 109.187 143.201 109.103 143.141L107.53 141.99L108.129 143.825C108.166 143.923 108.17 144.031 108.14 144.132C108.11 144.233 108.048 144.321 107.963 144.383L106.39 145.526H108.304C108.42 145.507 108.54 145.531 108.639 145.594C108.739 145.657 108.812 145.754 108.845 145.868V145.893Z"
                              fill="url(#paint41_linear_148_1506)"/>
                        <defs>
                            <linearGradient id="paint0_linear_148_1506" x1="121.768" y1="4.80301" x2="121.768"
                                            y2="28.7015" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint1_linear_148_1506" x1="125.122" y1="31.8702" x2="125.122"
                                            y2="35.706" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint2_linear_148_1506" x1="114.475" y1="45.212" x2="110.808"
                                            y2="19.7726" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint3_linear_148_1506" x1="102.637" y1="33.2461" x2="99.7401"
                                            y2="39.2795" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E64290"/>
                                <stop offset="1" stop-color="#571132"/>
                            </linearGradient>
                            <linearGradient id="paint4_linear_148_1506" x1="81.499" y1="26.5335" x2="85.4981"
                                            y2="36.7771" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint5_linear_148_1506" x1="91.9729" y1="40.0504" x2="91.9729"
                                            y2="44.9952" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint6_linear_148_1506" x1="82.9557" y1="53.5424" x2="72.1975"
                                            y2="33.9034" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#3F2671"/>
                                <stop offset="1" stop-color="#937CB6"/>
                            </linearGradient>
                            <linearGradient id="paint7_linear_148_1506" x1="83.5377" y1="55.7021" x2="96.1405"
                                            y2="70.3572" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E4E52B"/>
                                <stop offset="1" stop-color="#509036"/>
                            </linearGradient>
                            <linearGradient id="paint8_linear_148_1506" x1="51.7396" y1="46.196" x2="76.171"
                                            y2="65.6268" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint9_linear_148_1506" x1="58.1401" y1="64.6161" x2="78.0403"
                                            y2="82.1941" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#3BB9C5"/>
                                <stop offset="1" stop-color="#00516A"/>
                            </linearGradient>
                            <linearGradient id="paint10_linear_148_1506" x1="51.9129" y1="154.815" x2="54.1847"
                                            y2="135.153" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#937CB6"/>
                                <stop offset="1" stop-color="#3F2671"/>
                            </linearGradient>
                            <linearGradient id="paint11_linear_148_1506" x1="72.0911" y1="157.675" x2="86.0664"
                                            y2="143.882" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#3BB9C5"/>
                                <stop offset="1" stop-color="#00516A"/>
                            </linearGradient>
                            <linearGradient id="paint12_linear_148_1506" x1="87.467" y1="169.958" x2="65.7755"
                                            y2="176.301" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E7350F"/>
                                <stop offset="1" stop-color="#F1CB00"/>
                            </linearGradient>
                            <linearGradient id="paint13_linear_148_1506" x1="100.944" y1="171.642" x2="80.3241"
                                            y2="191.412" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#571132"/>
                                <stop offset="1" stop-color="#E64290"/>
                            </linearGradient>
                            <linearGradient id="paint14_linear_148_1506" x1="103.507" y1="189.587" x2="107.909"
                                            y2="164.924" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint15_linear_148_1506" x1="112.875" y1="193.79" x2="120.814"
                                            y2="174.103" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#937CB6"/>
                                <stop offset="1" stop-color="#3F2671"/>
                            </linearGradient>
                            <linearGradient id="paint16_linear_148_1506" x1="132.138" y1="164.488" x2="131.637"
                                            y2="200.411" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#509036"/>
                                <stop offset="1" stop-color="#E4E52B"/>
                            </linearGradient>
                            <linearGradient id="paint17_linear_148_1506" x1="143.135" y1="188.044" x2="143.143"
                                            y2="209.992" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E7350F"/>
                                <stop offset="1" stop-color="#F1CB00"/>
                            </linearGradient>
                            <linearGradient id="paint18_linear_148_1506" x1="149.054" y1="160.719" x2="155.46"
                                            y2="189.482" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint19_linear_148_1506" x1="169.006" y1="187.11" x2="171.203"
                                            y2="168.908" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E64290"/>
                                <stop offset="1" stop-color="#571132"/>
                            </linearGradient>
                            <linearGradient id="paint20_linear_148_1506" x1="182.254" y1="176.654" x2="180.508"
                                            y2="161.053" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint21_linear_148_1506" x1="195.831" y1="178.33" x2="205.661"
                                            y2="194.325" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#3F2671"/>
                                <stop offset="1" stop-color="#937CB6"/>
                            </linearGradient>
                            <linearGradient id="paint22_linear_148_1506" x1="205.978" y1="172.651" x2="190.563"
                                            y2="153.673" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E4E52B"/>
                                <stop offset="1" stop-color="#509036"/>
                            </linearGradient>
                            <linearGradient id="paint23_linear_148_1506" x1="232.333" y1="165.755" x2="202.929"
                                            y2="147.713" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint24_linear_148_1506" x1="210.957" y1="141.198" x2="233.776"
                                            y2="138.248" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint25_linear_148_1506" x1="216.335" y1="67.5179" x2="227.212"
                                            y2="54.8195" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint26_linear_148_1506" x1="215.218" y1="43.3025" x2="209.087"
                                            y2="50.889" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E64290"/>
                                <stop offset="1" stop-color="#571132"/>
                            </linearGradient>
                            <linearGradient id="paint27_linear_148_1506" x1="205.779" y1="31.9036" x2="191.905"
                                            y2="50.8852" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint28_linear_148_1506" x1="185.277" y1="37.9991" x2="185.285"
                                            y2="14.4425" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#3F2671"/>
                                <stop offset="1" stop-color="#937CB6"/>
                            </linearGradient>
                            <linearGradient id="paint29_linear_148_1506" x1="169.824" y1="27.5008" x2="163.095"
                                            y2="44.4788" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E4E52B"/>
                                <stop offset="1" stop-color="#509036"/>
                            </linearGradient>
                            <linearGradient id="paint30_linear_148_1506" x1="156.771" y1="20.7965" x2="156.771"
                                            y2="38.7496" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint31_linear_148_1506" x1="146.898" y1="18.6118" x2="147.224"
                                            y2="8.9724" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint32_linear_148_1506" x1="144.631" y1="25.8414" x2="145.792"
                                            y2="31.0022" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E64290"/>
                                <stop offset="1" stop-color="#571132"/>
                            </linearGradient>
                            <linearGradient id="paint33_linear_148_1506" x1="135.719" y1="17.5612" x2="131.898"
                                            y2="18.251" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint34_linear_148_1506" x1="139.198" y1="38.908" x2="131.818"
                                            y2="46.3921" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#E4E52B"/>
                                <stop offset="1" stop-color="#509036"/>
                            </linearGradient>
                            <linearGradient id="paint35_linear_148_1506" x1="117.079" y1="64.9495" x2="117.079"
                                            y2="80.7762" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint36_linear_148_1506" x1="237.388" y1="70.5031" x2="219.181"
                                            y2="82.2494" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint37_linear_148_1506" x1="150.537" y1="64.9162" x2="150.62"
                                            y2="55.4435" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint38_linear_148_1506" x1="184.66" y1="60.6885" x2="183.682"
                                            y2="69.2936" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                            <linearGradient id="paint39_linear_148_1506" x1="178.598" y1="145.442" x2="178.598"
                                            y2="129.607" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#F1CB00"/>
                                <stop offset="1" stop-color="#E7350F"/>
                            </linearGradient>
                            <linearGradient id="paint40_linear_148_1506" x1="143.311" y1="140.889" x2="143.236"
                                            y2="150.362" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#00516A"/>
                                <stop offset="1" stop-color="#3BB9C5"/>
                            </linearGradient>
                            <linearGradient id="paint41_linear_148_1506" x1="109.087" y1="148.953" x2="110.072"
                                            y2="140.348" gradientUnits="userSpaceOnUse">
                                <stop stop-color="#91C31E"/>
                                <stop offset="1" stop-color="#41602F"/>
                            </linearGradient>
                        </defs>
                    </svg>
                </div>
                <h2><?php echo esc_html__( 'Congratulation', 'designbox-builder' ); ?></h2>
                <div class="desc"> <?php echo esc_html__( 'You have completed your setup for "DesignBox Builder" add more creativity to your design with "DesignBox Builder".', 'designbox-builder' ); ?></div>
                <button type="button"
                        class="wdb-admin-btn wdb-settings-save"><?php esc_html_e( 'Save Settings', 'designbox-builder' ); ?></button>
            </div>
        </div>
		<?php
	}

	/**
	 * [remove_all_notices] remove addmin notices
	 * @return [void]
	 */
	public function remove_all_notices() {
		add_action( 'in_admin_header', function () {
			if ( isset( $_GET['page'] ) && $_GET['page'] == 'wdb_addons_setup_page' ) {
				remove_all_actions( 'admin_notices' );
				remove_all_actions( 'all_admin_notices' );
			}
		}, 1000 );
	}

}

WDB_Setup_Wizard_Init::instance();

