<?php

namespace WDB_ADDONS;

use Elementor\Modules\Library\Documents\Library_Document;
use Elementor\Plugin as ElementorPlugin;

if ( ! defined( 'ABSPATH' ) ) {
	exit();
} // Exit if accessed directly

class WDB_Theme_Builder {

	const CPTTYPE = 'wdb-addons-template';
	const CPT_META = 'wdb-addons-template-meta';


	/**
	 * [$_instance]
	 * @var null
	 */
	public static $_instance = null;

	/**
	 * [instance] Initializes a singleton instance
	 * @return [Woolentor_Admin_Init]
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function __construct() {

		add_action( 'init', [ $this, 'init' ] );

		//Add Menu
		add_action( 'admin_menu', [ $this, 'admin_menu' ], 225 );

		// Print template tabs.
		add_filter( 'views_edit-' . self::CPTTYPE, [ $this, 'print_tabs' ] );

		// query filter
		add_filter( 'parse_query', [ $this, 'query_filter' ] );

		// Template type column.
		add_action( 'manage_' . self::CPTTYPE . '_posts_columns', [ $this, 'manage_columns' ] );
		add_action( 'manage_' . self::CPTTYPE . '_posts_custom_column', [ $this, 'columns_content' ], 10, 2 );

		// Print template edit popup.
		add_action( 'admin_footer', [ $this, 'print_popup' ] );

		// Template store ajax action
		add_action( 'wp_ajax_wdb_save_template', [ $this, 'save_template_request' ] );

		// Get template data Ajax action
		add_action( 'wp_ajax_wdb_get_template', [ $this, 'get_post_By_id' ] );

		add_action( 'wp_ajax_wdb_get_posts_by_query', array( $this, 'get_posts_by_query' ) );

		// Load Scripts
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

		// Change Template
		add_filter( 'template_include', [ $this, 'template_loader' ] );

		// Archive Page
		add_action( 'wdb_archive_builder_content', [ $this, 'archive_page_builder_content' ] );

		//single
		add_action( 'wdb_single_builder_content', [ $this, 'single_post_builder_content' ] );

		// Body classes
		add_filter( 'body_class', [ $this, 'body_classes' ] );

		//header footer
		add_action( 'get_header', [ $this, 'override_header' ] );
		add_action( 'get_footer', [ $this, 'override_footer' ] );
		add_action( 'wdb_header_builder_content', [ $this, 'header_builder_content' ] );
		add_action( 'wdb_footer_builder_content', [ $this, 'footer_builder_content' ] );
	}

	/**
	 * @param $post_id
	 *
	 * @return Library_Document
	 */
	public static function get_document( $post_id ) {
		$document = null;

		try {
			$document = ElementorPlugin::$instance->documents->get( $post_id );
		} catch ( \Exception $e ) {
			// Do nothing.
			unset( $e );
		}

		if ( ! empty( $document ) && ! $document instanceof Library_Document ) {
			$document = null;
		}

		return $document;
	}

	//header footer//

	/**
	 * Function for overriding the header in the elmentor way.
	 *
	 *
	 */
	public function override_header( $name ) {

		if ( ! $this->has_template( 'header' ) ) {
			return;
		}

		require WDB_ADDONS_PATH . '/templates/header.php';

		$templates = array();
		$name      = (string) $name;
		if ( '' !== $name ) {
			$templates[] = "header-{$name}.php";
		}

		$templates[] = 'header.php';

		// Avoid running wp_head hooks again
		remove_all_actions( 'wp_head' );
		ob_start();
		// It cause a `require_once` so, in the get_header it self it will not be required again.
		locate_template( $templates, true );
		ob_get_clean();
	}

	/**
	 * Function for overriding the footer in the elmentor way.
	 *
	 */
	public function override_footer( $name ) {

		//fixed div ending issues
		if ( ! $this->has_template( 'header' ) && $this->has_template( 'footer' ) ) {

			$current_template = basename( get_page_template_slug() );
			if ( $current_template == 'elementor_canvas' ) {
				return;
			}

			$current_theme = get_template();

			switch ( $current_theme ) {
				case 'astra':
					echo '</div></div>';
					break;

//				case 'neve':
//					echo '</main>';
//					break;
//
//				case 'generatepress':
//				case 'generatepress-child':
//				    echo '</div>';
//					break;
//
//				case 'oceanwp':
//				case 'oceanwp-child':
//				    echo '</div>';
//					break;
//
//				case 'bb-theme':
//				case 'bb-theme-child':
//				        echo '</div>';
//					break;
//
//				case 'genesis':
//				case 'genesis-child':
//				    echo '</div>';
//					break;
//
//				case 'twentynineteen':
//					echo '</div>';
//					break;
//
//				case 'my-listing':
//				case 'my-listing-child':
//				    echo '</div>';
//					break;

				default:
					break;
			}
		}


		if ( ! $this->has_template( 'footer' ) ) {
			return;
		}

		require WDB_ADDONS_PATH . '/templates/footer.php';

		$templates = array();
		$name      = (string) $name;
		if ( '' !== $name ) {
			$templates[] = "footer-{$name}.php";
		}

		$templates[] = 'footer.php';

		// Avoid running wp_head hooks again
		remove_all_actions( 'wp_footer' );
		ob_start();
		// It cause a `require_once` so, in the get_header it self it will not be required again.
		locate_template( $templates, true );
		ob_get_clean();
	}

	// Set Builder content header footer
	public function header_builder_content() {

		$archive_template_id = $this->get_template_id( 'header' );
		if ( $archive_template_id != '0' ) {
			// PHPCS - should not be escaped.
			echo self::render_build_content( $archive_template_id ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	public function footer_builder_content() {
		$archive_template_id = $this->get_template_id( 'footer' );
		if ( $archive_template_id != '0' ) {
			// PHPCS - should not be escaped.
			echo self::render_build_content( $archive_template_id ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}
	//header footer//


	///template///
	/**
	 * [body_classes]
	 *
	 * @param  [array] $classes
	 *
	 * @return [array]
	 */
	public function body_classes( $classes ) {

		$class_prefix = 'elementor-page-';

		if ( is_singular() && $this->has_template( 'single' ) ) {
			$classes[] = $class_prefix . self::has_template( 'single' );
		} elseif ( ( is_archive() || is_home() || is_search() ) && $this->has_template( 'archive' ) ) {
			$classes[] = $class_prefix . self::has_template( 'archive' );
		}

		return $classes;

	}

	/**
	 * Load template.
	 *
	 * @param string $template template to load.
	 *
	 * @return string
	 */
	public function template_loader( $template ) {

		if ( is_embed() ) {
			return $template;
		}

		$default_file = self::get_template_loader_default_file();

		if ( $default_file ) {
			$template = WDB_ADDONS_PATH . '/templates/' . $default_file;
		}

		return $template;

	}

	/**
	 * Get the default filename for a template except if a block template with
	 * the same name exists.
	 *
	 * @return string
	 * @since  5.5.0 If a block template with the same name exists, return an
	 * empty string.
	 * @since  6.3.0 It checks custom product taxonomies
	 * @since  3.0.0
	 */
	private function get_template_loader_default_file() {
		if ( is_singular() && $this->has_template( 'single' ) ) {
			$default_file = 'single.php';
		} elseif ( ( is_archive() || is_home() || is_search() || is_404() ) && $this->has_template( 'archive' ) ) {
			$default_file = 'archive.php';
		} else {
			$default_file = '';
		}

		return $default_file;
	}

	/**
	 * [has_template]
	 *
	 * @param  [string]  $field_key
	 *
	 * @return boolean | int
	 */
	public function has_template( $tmpType = '' ) {
		$template_ID = self::get_current_post_by_condition( $tmpType );

		if ( $template_ID ) {
			return $template_ID;
		}

		return false;
	}

	/**
	 * [get_template_id]
	 *
	 * @param  [string]  $field_key
	 * @param  [string]  $meta_key
	 *
	 * @return boolean | int
	 */
	public function get_template_id( $tmpType = '' ) {
		$template_ID = self::get_current_post_by_condition( $tmpType );

		if ( $template_ID ) {
			return $template_ID;
		}

		return false;
	}

	public function get_current_post_by_condition( $tmpType = '' ) {
		$query_args         = [
			'post_type'      => self::CPTTYPE,
			'fields'         => 'ids',
			'posts_per_page' => - 1,
			'order'          => 'ASC',
			'orderby'        => 'date',
			'meta_query'     => array(
				array(
					'key'   => self::CPT_META . '_type',
					'value' => $tmpType,
				)
			)
		];
		$query              = new \WP_Query( $query_args );
		$count              = $query->post_count;
		$templates          = [];
		$templates_specific = [ 'specifics' => [] ];

		foreach ( $query->posts as $key => $post_id ) {

			$location   = get_post_meta( absint( $post_id ), self::CPT_META . '_location', true );
			$splocation = get_post_meta( absint( $post_id ), self::CPT_META . '_splocation', true );

			if ( ! empty( $location ) ) {
				if ( 'specifics' === $location ) {
					array_push( $templates_specific['specifics'],
						[ 'id' => $post_id, 'posts' => json_decode( $splocation ) ]
					);
				} else {
					$templates[ $location ] = $post_id;
				}
			}

			if ( $key === $count - 1 && ! empty( $templates_specific['specifics'] ) ) {
				$templates = array_merge( $templates, $templates_specific );
			}
		}

		wp_reset_postdata();

		if ( empty( $templates ) ) {
			return false;
		}

		//check for specific page and post
		if ( ! is_home() && ! is_archive() && array_key_exists( 'specifics', $templates ) ) {
			foreach ( $templates['specifics'] as $specific ) {
				$key = array_search( get_the_ID(), $specific['posts'] );
				if ( false !== $key ) {
					return $specific['id'];
				}
			}
		}

		//check 404 page
		if ( is_404() && array_key_exists( '404', $templates ) ) {
			return $templates['404'];
		}

		//check search page
		if ( is_search() && array_key_exists( 'search', $templates ) ) {
			return $templates['search'];
		}

		//check front page
		if ( is_front_page() && array_key_exists( 'front', $templates ) ) {
			return $templates['front'];
		}

		//check for blog/posts page
		if ( is_home() && array_key_exists( 'blog', $templates ) ) {
			return $templates['blog'];
		}

		//check for archive
		if ( is_archive() ) {

			//check for all date archive
			if ( is_date() && array_key_exists( 'date', $templates ) ) {
				return $templates['date'];
			}

			//check for all author archive
			if ( is_author() && array_key_exists( 'author', $templates ) ) {
				return $templates['author'];
			}

			//check for WooCommerce shop archive
			if ( function_exists( 'is_shop' ) && is_shop() && array_key_exists( 'woo-shop', $templates ) ) {
				return $templates['woo-shop'];
			}

			//check for custom post type archive
			$custom_archive = get_post_type() . '-archive';
			if ( array_key_exists( $custom_archive, $templates ) ) {
				return $templates[ $custom_archive ];
			}

			//all archives
			if ( array_key_exists( 'archives', $templates ) ) {
				return $templates['archives'];
			}
		}

		//check for singular
		if ( is_singular() ) {

			//if template type single ignore post type page
			if ( ( 'page' === get_post_type() || self::CPTTYPE === get_post_type() ) && 'single' === $tmpType ) {
				return false;
			}

			//check for custom post type singular
			$custom_single = get_post_type() . '-singular';
			if ( array_key_exists( $custom_single, $templates ) ) {
				return $templates[ $custom_single ];
			}

			//all singular
			if ( array_key_exists( 'singulars', $templates ) ) {
				return $templates['singulars'];
			}
		}


		//check for global
		if ( array_key_exists( 'global', $templates ) ) {
			return $templates['global'];
		}
	}

	/**
	 * Get current page type
	 *
	 * @return string Page Type.
	 * @since  1.0.0
	 *
	 */
	public function get_current_page_type() {
		$page_type = '';

		if ( is_404() ) {
			$page_type = 'is_404';
		} elseif ( is_search() ) {
			$page_type = 'is_search';
		} elseif ( is_archive() ) {
			$page_type = 'is_archive';

			if ( is_category() || is_tag() || is_tax() ) {
				$page_type = 'is_tax';
			} elseif ( is_date() ) {
				$page_type = 'is_date';
			} elseif ( is_author() ) {
				$page_type = 'is_author';
			} elseif ( function_exists( 'is_shop' ) && is_shop() ) {
				$page_type = 'is_woo_shop_page';
			}
		} elseif ( is_home() ) {
			$page_type = 'is_home';
		} elseif ( is_front_page() ) {
			$page_type  = 'is_front_page';
			$current_id = get_the_id();
		} elseif ( is_singular() ) {
			$page_type  = 'is_singular';
			$current_id = get_the_id();
		} else {
			$current_id = get_the_id();
		}

		return $page_type;
	}

	/**
	 * [render_build_content]
	 *
	 * @param  [int]  $id
	 *
	 * @return string
	 */
	public static function render_build_content( $id ) {
		$output   = '';
		$document = ElementorPlugin::instance()->documents->get( $id );

		if ( $document && $document->is_built_with_elementor() ) {
			$output = ElementorPlugin::instance()->frontend->get_builder_content_for_display( $id, true );
		} else {
			$content = get_the_content( null, false, $id );

			if ( has_blocks( $content ) ) {
				$blocks = parse_blocks( $content );
				foreach ( $blocks as $block ) {
					$output .= do_shortcode( render_block( $block ) );
				}
			} else {
				$content = apply_filters( 'the_content', $content );
				$content = str_replace( ']]>', ']]&gt;', $content );

				return $content;
			}
		}

		return $output;

	}

	// Set Builder Content For archive page
	public function archive_page_builder_content() {
		$archive_template_id = $this->get_template_id( 'archive' );
		if ( $archive_template_id != '0' ) {
			// PHPCS - should not be escaped.
			echo self::render_build_content( $archive_template_id );// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}

	// Set Builder Content For single post
	public function single_post_builder_content() {
		$archive_template_id = $this->get_template_id( 'single' );
		if ( $archive_template_id != '0' ) {
			// PHPCS - should not be escaped.
			echo self::render_build_content( $archive_template_id ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
	}
	///template///


	/**
	 * Print Admin Tab
	 *
	 * @param [array] $views
	 *
	 * @return array
	 */
	public function print_tabs( $views ) {
		$active_class = 'nav-tab-active';
		$current_type = '';
		if ( isset( $_GET['template_type'] ) ) {
			$active_class = '';
			$current_type = sanitize_key( $_GET['template_type'] );
		}
		?>
        <div id="wdb-template-tabs-wrapper" class="nav-tab-wrapper">
            <div class="wdb-menu-area">
                <a class="nav-tab <?php echo esc_attr( $active_class ); ?>"
                   href="edit.php?post_type=<?php echo esc_attr( self::CPTTYPE ); ?>">
					<?php echo esc_html__( 'All', 'designbox-builder' ); ?>
                </a>
				<?php
				foreach ( self::get_template_type() as $tabkey => $tab ) {
					$active_class = ( $current_type == $tabkey ? 'nav-tab-active' : '' );
					$url          = 'edit.php?post_type=' . self::CPTTYPE . '&template_type=' . $tabkey;

					printf( '<a class="nav-tab %s" href="%s">%s</a>',
						esc_attr( $active_class ),
						esc_url( $url ),
						esc_html( $tab['label'] )
					);
				}
				?>
            </div>
        </div>
		<?php
		return $views;
	}

	/**
	 * Manage Template filter by template type
	 *
	 * @param \WP_Query $query
	 *
	 * @return void
	 */
	public function query_filter( \WP_Query $query ) {
		if ( ! is_admin() || ! empty( $query->query_vars['meta_key'] ) || self::CPTTYPE !== $query->get( 'post_type' ) ) {
			return;
		}

		if ( isset( $_GET['template_type'] ) && $_GET['template_type'] != '' && $_GET['template_type'] != 'all' ) {
			$type = isset( $_GET['template_type'] ) ? sanitize_key( $_GET['template_type'] ) : '';

			$query->query_vars['meta_key']     = self::CPT_META . '_type';
			$query->query_vars['meta_value']   = $type;
			$query->query_vars['meta_compare'] = '=';
		}
	}

	/**
	 * Manage Post Table columns
	 *
	 * @param [array] $columns
	 *
	 * @return array
	 */
	public function manage_columns( $columns ) {

		$column_date   = $columns['date'];
		unset( $columns['date'] );

		$columns['type']   = esc_html__( 'Type', 'designbox-builder' );
		$columns['status'] = esc_html__( 'Display', 'designbox-builder' );
		$columns['date']   = esc_html( $column_date );

		return $columns;
	}

	/**
	 * Manage Custom column content
	 *
	 * @param [string] $column_name
	 * @param [int] $post_id
	 *
	 * @return void
	 */
	public function columns_content( $column_name, $post_id ) {
		$tmpType = get_post_meta( $post_id, self::CPT_META . '_type', true );

		if ( ! array_key_exists( $tmpType, self::get_template_type() ) ) {
			return;
		}

		if ( $column_name === 'type' ) {
			// PHPCS - should not be escaped.
			echo isset( self::get_template_type()[ $tmpType ] ) ? '<div class="column-tmptype">' . self::get_template_type()[ $tmpType ]['label'] . '</div>' : '-'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}

		if ( $column_name === 'status' ) {
			$tmpDisplay = get_post_meta( $post_id, self::CPT_META . '_location', true );
			?>
            <div class="post-status">
            <strong>Display: </strong>
			<?php echo esc_html( $tmpDisplay ); ?>
            </div><?php
		}

	}

	/**
	 * Get Template Type
	 *
	 * @return array
	 */
	public static function get_template_type() {

		$template_type = [
			'header'  => [
				'label'     => esc_html__( 'Header', 'designbox-builder' ),
				'optionkey' => 'header'
			],
			'footer'  => [
				'label'     => esc_html__( 'Footer', 'designbox-builder' ),
				'optionkey' => 'footer'
			],
			'archive' => [
				'label'     => esc_html__( 'Archive/404/Search', 'designbox-builder' ),
				'optionkey' => 'archivepage'
			],
			'single'  => [
				'label'     => esc_html__( 'Single', 'designbox-builder' ),
				'optionkey' => 'singlepage'
			],
		];

		return apply_filters( 'wdb_builder_template_types', $template_type );

	}

	/**
	 * Get header footer location selection options.
	 *
	 * @return array
	 */
	public static function get_hf_location_selections() {
		$args = array(
			'public'            => true,
			'show_in_nav_menus' => true,
		);

		$post_types = get_post_types( $args, 'objects' );

		$special_pages = array(
			'404'    => esc_html__( '404 Page', 'designbox-builder' ),
			'search' => esc_html__( 'Search Page', 'designbox-builder' ),
			'blog'   => esc_html__( 'Blog / Posts Page', 'designbox-builder' ),
			'front'  => esc_html__( 'Front Page', 'designbox-builder' ),
			'date'   => esc_html__( 'Date Archive', 'designbox-builder' ),
			'author' => esc_html__( 'Author Archive', 'designbox-builder' ),
		);

		if ( class_exists( 'WooCommerce' ) ) {
			$special_pages['woo-shop'] = esc_html__( 'WooCommerce Shop Page', 'designbox-builder' );
		}

		$selection_options = array(
			'basic' => array(
				'label' => esc_html__( 'Basic', 'designbox-builder' ),
				'value' => array(
					''          => esc_html__( 'None', 'designbox-builder' ),
					'global'    => esc_html__( 'Entire Website', 'designbox-builder' ),
					'singulars' => esc_html__( 'All Singulars', 'designbox-builder' ),
					'archives'  => esc_html__( 'All Archives', 'designbox-builder' ),
				),
			),

			'special-pages' => array(
				'label' => esc_html__( 'Special Pages', 'designbox-builder' ),
				'value' => $special_pages,
			),
		);


		foreach ( $post_types as $post_type ) {
			if ( 'page' === $post_type->name ) {
				$selection_options[ $post_type->name ] = [
					'label' => esc_html( $post_type->label ),
					'value' => array(
						'all' . $post_type->name => esc_html( 'All' . $post_type->label ),
					),
				];
			} else {
				$selection_options[ $post_type->name ] = [
					'label' => esc_html( $post_type->label ),
					'value' => array(
						$post_type->name . '-archive'   => esc_html( $post_type->label . ' Archive' ),
						$post_type->name . '-singulars' => esc_html( $post_type->label . ' Singulars' ),
					),
				];
			}
		}


		$selection_options['specific-target'] = array(
			'label' => esc_html__( 'Specific Target', 'designbox-builder' ),
			'value' => array(
				'specifics' => esc_html__( 'Specific Pages / Posts.', 'designbox-builder' ),
			),
		);

		/**
		 * Filter options displayed in the display conditions select field of Display conditions.
		 *
		 * @since 1.0.0
		 */
		return apply_filters( 'wdb_display_hf_list', $selection_options );
	}

	/**
	 * Get archive location selection options.
	 *
	 * @return array
	 */
	public static function get_archive_location_selections() {
		$args = array(
			'public'            => true,
			'show_in_nav_menus' => true,
		);

		$post_types = get_post_types( $args, 'objects' );

		//unset unnecessary post type
		unset( $post_types['page'] );
		unset( $post_types['post'] );
		unset( $post_types['product'] );
		unset( $post_types[ self::CPTTYPE ] );

		$special_pages = array(
			'404'    => esc_html__( '404 Page', 'designbox-builder' ),
			'search' => esc_html__( 'Search Page', 'designbox-builder' ),
			'blog'   => esc_html__( 'Blog / Posts Page', 'designbox-builder' ),
			'date'   => esc_html__( 'Date Archive', 'designbox-builder' ),
			'author' => esc_html__( 'Author Archive', 'designbox-builder' ),
		);

		$selection_options = array(
			'basic' => array(
				'label' => esc_html__( 'Basic', 'designbox-builder' ),
				'value' => array(
					''         => esc_html__( 'None', 'designbox-builder' ),
					'archives' => esc_html__( 'All Archives', 'designbox-builder' ),
				),
			),

			'special-pages' => array(
				'label' => esc_html__( 'Special Pages', 'designbox-builder' ),
				'value' => $special_pages,
			),
		);


		foreach ( $post_types as $post_type ) {
			$selection_options[ $post_type->name ] = [
				'label' => esc_html( $post_type->label ),
				'value' => array(
					$post_type->name . '-archive' => esc_html( $post_type->label . ' Archive' ),
				),
			];
		}

		/**
		 * Filter options displayed in the display conditions select field of Display conditions.
		 *
		 * @since 1.0.0
		 */
		return apply_filters( 'wdb_display_archive_list', $selection_options );
	}

	/**
	 * Get single location selection options.
	 *
	 * @return array
	 */
	public static function get_single_location_selections() {
		$args = array(
			'public'            => true,
			'show_in_nav_menus' => true,
		);

		$post_types = get_post_types( $args, 'objects' );

		//unset unnecessary post type
		unset( $post_types['page'] );
		unset( $post_types['product'] );
		unset( $post_types[ self::CPTTYPE ] );

		$selection_options = array(
			'basic' => array(
				'label' => esc_html__( 'Basic', 'designbox-builder' ),
				'value' => array(
					'singulars' => esc_html__( 'All Singular', 'designbox-builder' ),
				),
			),
		);


		foreach ( $post_types as $post_type ) {
			$selection_options[ $post_type->name ] = [
				'label' => esc_html( $post_type->label, 'designbox-builder' ),
				'value' => array(
					$post_type->name . '-singular' => esc_html( $post_type->label . ' Singular', 'designbox-builder' ),
				),
			];
		}

		/**
		 * Filter options displayed in the display conditions select field of Display conditions.
		 *
		 * @since 1.0.0
		 */
		return apply_filters( 'wdb_display_archive_list', $selection_options );
	}

	/**
	 * Register Builder Custom post
	 *
	 * @return void
	 */
	public function register_custom_post_type() {

		$labels = array(
			'name'                  => esc_html_x( 'DesignBox Builder', 'Post Type General Name', 'designbox-builder' ),
			'singular_name'         => esc_html_x( 'DesignBox Builder', 'Post Type Singular Name', 'designbox-builder' ),
			'menu_name'             => esc_html__( 'DesignBox Builder', 'designbox-builder' ),
			'name_admin_bar'        => esc_html__( 'DesignBox Builder', 'designbox-builder' ),
			'archives'              => esc_html__( 'Template Archives', 'designbox-builder' ),
			'attributes'            => esc_html__( 'Template Attributes', 'designbox-builder' ),
			'parent_item_colon'     => esc_html__( 'Parent Item:', 'designbox-builder' ),
			'all_items'             => esc_html__( 'Templates', 'designbox-builder' ),
			'add_new_item'          => esc_html__( 'Add New Template', 'designbox-builder' ),
			'add_new'               => esc_html__( 'Add New', 'designbox-builder' ),
			'new_item'              => esc_html__( 'New Template', 'designbox-builder' ),
			'edit_item'             => esc_html__( 'Edit Template', 'designbox-builder' ),
			'update_item'           => esc_html__( 'Update Template', 'designbox-builder' ),
			'view_item'             => esc_html__( 'View Template', 'designbox-builder' ),
			'view_items'            => esc_html__( 'View Templates', 'designbox-builder' ),
			'search_items'          => esc_html__( 'Search Templates', 'designbox-builder' ),
			'not_found'             => esc_html__( 'Not found', 'designbox-builder' ),
			'not_found_in_trash'    => esc_html__( 'Not found in Trash', 'designbox-builder' ),
			'featured_image'        => esc_html__( 'Featured Image', 'designbox-builder' ),
			'set_featured_image'    => esc_html__( 'Set featured image', 'designbox-builder' ),
			'remove_featured_image' => esc_html__( 'Remove featured image', 'designbox-builder' ),
			'use_featured_image'    => esc_html__( 'Use as featured image', 'designbox-builder' ),
			'insert_into_item'      => esc_html__( 'Insert into Template', 'designbox-builder' ),
			'uploaded_to_this_item' => esc_html__( 'Uploaded to this Template', 'designbox-builder' ),
			'items_list'            => esc_html__( 'Templates list', 'designbox-builder' ),
			'items_list_navigation' => esc_html__( 'Templates list navigation', 'designbox-builder' ),
			'filter_items_list'     => esc_html__( 'Filter from list', 'designbox-builder' ),
		);

		$args = array(
			'label'               => esc_html__( 'Theme Builder', 'designbox-builder' ),
			'description'         => esc_html__( ' Theme Builder', 'designbox-builder' ),
			'labels'              => $labels,
			'supports'            => array( 'title', 'elementor', 'thumbnail' ),
			'hierarchical'        => false,
			'public'              => true,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'show_in_admin_bar'   => false,
			'show_in_nav_menus'   => true,
			'can_export'          => true,
			'has_archive'         => false,
			'rewrite'             => array(
				'slug'       => 'wdb-template',
				'pages'      => false,
				'with_front' => true,
				'feeds'      => false,
			),
			'query_var'           => true,
			'exclude_from_search' => true,
			'publicly_queryable'  => true,
			'capability_type'     => 'page',
			'show_in_rest'        => true,
			'rest_base'           => self::CPTTYPE,
		);

		register_post_type( self::CPTTYPE, $args );

		flush_rewrite_rules();
	}

	/**
	 * Print Template edit popup
	 *
	 * @return void
	 */
	public function print_popup() {
		if ( isset( $_GET['post_type'] ) && $_GET['post_type'] == self::CPTTYPE ) {
			?>
            <script type="text/template" id="tmpl-wdb-addons-ctppopup">
                <div class="wdb-addons-template-edit-popup-area">
                    <div class="wdb-addons-body-overlay"></div>
                    <div class="wdb-addons-template-edit-popup">

                        <div class="wdb-addons-template-edit-header">
                            <h3 class="wdb-addons-template-edit-setting-title">
                                {{{data.heading.head}}}
                            </h3>
                            <span class="wdb-addons-template-edit-cross">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor"
                                     class="bi bi-x-lg" viewBox="0 0 16 16"><path
                                            d="M2.146 2.854a.5.5 0 1 1 .708-.708L8 7.293l5.146-5.147a.5.5 0 0 1 .708.708L8.707 8l5.147 5.146a.5.5 0 0 1-.708.708L8 8.707l-5.146 5.147a.5.5 0 0 1-.708-.708L7.293 8z"/></svg>
                            </span>
                        </div>

                        <div class="wdb-addons-template-edit-body">

                            <div class="wdb-addons-template-edit-field">
                                <label class="wdb-addons-template-edit-label">{{{ data.heading.fields.name.title
                                    }}}</label>
                                <input class="wdb-addons-template-edit-input" id="wdb-addons-template-title" type="text"
                                       name="wdb-addons-template-title"
                                       placeholder="{{ data.heading.fields.name.placeholder }}">
                            </div>

                            <div class="wdb-addons-template-edit-field">
                                <label class="wdb-addons-template-edit-label">{{{data.heading.fields.type}}}</label>
                                <select class="wdb-addons-template-edit-input" name="wdb-addons-template-type"
                                        id="wdb-addons-template-type">
                                    <#
                                    _.each( data.templatetype, function( item, key ) {

                                    #>
                                    <option value="{{ key }}">{{{ item.label }}}</option>
                                    <#

                                    } );
                                    #>
                                </select>
                            </div>

                            <div class="wdb-addons-template-edit-field hf-location hidden">
                                <label class="wdb-addons-template-edit-label">{{{data.heading.fields.display}}}</label>
                                <select class="wdb-addons-template-edit-input" name="wdb-addons-hf-display-type"
                                        id="wdb-addons-hf-display-type">
                                    <#
                                    _.each( data.hflocation, function( items, keys ) {
                                    #>
                                    <optgroup label="{{{ items.label }}}">
                                        <#
                                        _.each( items.value, function( item, key ) {
                                        #>
                                        <option value="{{ key }}">{{{ item }}}</option>
                                        <#
                                        } );
                                        #>
                                    </optgroup>
                                    <#
                                    } );
                                    #>
                                </select>
                            </div>

                            <div class="wdb-addons-template-edit-field hf-s-location hidden">
                                <label class="wdb-addons-template-edit-label"></label>
                                <select class="wdb-addons-template-edit-input" name="wdb-addons-hf-s-display-type[]"
                                        id="wdb-addons-hf-s-display-type" multiple="multiple">
                                </select>
                            </div>

                            <div class="wdb-addons-template-edit-field archive-location hidden">
                                <label class="wdb-addons-template-edit-label">{{{data.heading.fields.display}}}</label>
                                <select class="wdb-addons-template-edit-input" name="wdb-addons-archive-display-type"
                                        id="wdb-addons-archive-display-type">
                                    <#
                                    _.each( data.archivelocation, function( items, keys ) {
                                    #>
                                    <optgroup label="{{{ items.label }}}">
                                        <#
                                        _.each( items.value, function( item, key ) {
                                        #>
                                        <option value="{{ key }}">{{{ item }}}</option>
                                        <#
                                        } );
                                        #>
                                    </optgroup>
                                    <#
                                    } );
                                    #>
                                </select>
                            </div>

                            <div class="wdb-addons-template-edit-field single-location hidden">
                                <label class="wdb-addons-template-edit-label">{{{data.heading.fields.display}}}</label>
                                <select class="wdb-addons-template-edit-input" name="wdb-addons-single-display-type"
                                        id="wdb-addons-single-display-type">
                                    <#
                                    _.each( data.singlelocation, function( items, keys ) {
                                    #>
                                    <optgroup label="{{{ items.label }}}">
                                        <#
                                        _.each( items.value, function( item, key ) {
                                        #>
                                        <option value="{{ key }}">{{{ item }}}</option>
                                        <#
                                        } );
                                        #>
                                    </optgroup>
                                    <#
                                    } );
                                    #>
                                </select>
                            </div>

                        </div>

                        <div class="wdb-addons-template-edit-footer">

                            <div class="wdb-addons-template-button-group">
                                <div class="wdb-addons-template-button-item wdb-addons-editor-elementor {{ data.haselementor === 'yes' ? 'button-show' : '' }}">
                                    <button class="wdb-addons-tmp-elementor button">{{{
                                        data.heading.buttons.elementor.label
                                        }}}
                                    </button>
                                </div>
                                <div class="wdb-addons-template-button-item">
                                    <button class="wdb-addons-tmp-save button button-primary">{{{
                                        data.heading.buttons.save.label }}}
                                    </button>
                                </div>
                            </div>

                        </div>

                    </div>
                </div>
            </script>
			<?php
		}
	}

	/**
	 * Save Template
	 *
	 * @return void
	 */
	public function save_template_request() {
		if ( isset( $_POST ) ) {

			if ( ! ( current_user_can( 'manage_options' ) || current_user_can( 'edit_others_posts' ) ) ) {
				$errormessage = array(
					'message' => esc_html__( 'You are unauthorize to adding template!', 'designbox-builder' )
				);
				wp_send_json_error( $errormessage );
			}

			$nonce = $_POST['nonce'];
			if ( ! wp_verify_nonce( $nonce, 'wdb_tmp_nonce' ) ) {
				$errormessage = array(
					'message' => esc_html__( 'Nonce Varification Faild !', 'designbox-builder' )
				);
				wp_send_json_error( $errormessage );
			}

			$title            = ! empty( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
			$tmpid            = ! empty( $_POST['tmpId'] ) ? sanitize_text_field( $_POST['tmpId'] ) : '';
			$tmpType          = ! empty( $_POST['tmpType'] ) ? sanitize_text_field( $_POST['tmpType'] ) : 'single';
			$tmplocation      = ! empty( $_POST['tmpDisplay'] ) ? sanitize_text_field( $_POST['tmpDisplay'] ) : '';
			$specificsDisplay = ! empty( $_POST['specificsDisplay'] ) ? sanitize_text_field( $_POST['specificsDisplay'] ) : '';

			$data = [
				'title'         => $title,
				'id'            => $tmpid,
				'tmptype'       => $tmpType,
				'tmplocation'   => $tmplocation,
				'tmpSpLocation' => $specificsDisplay,
			];


			if ( $tmpid ) {
				$this->update( $data );
			} else {
				$this->insert( $data );
			}

		} else {
			$errormessage = array(
				'message' => esc_html__( 'Post request dose not found', 'designbox-builder' )
			);
			wp_send_json_error( $errormessage );
		}

	}

	/**
	 * Get Template data by id
	 *
	 * @return void
	 */
	public function get_post_By_id() {
		if ( isset( $_POST ) ) {

			if ( ! ( current_user_can( 'manage_options' ) || current_user_can( 'edit_others_posts' ) ) ) {
				$errormessage = array(
					'message' => esc_html__( 'You are unauthorize to adding template!', 'designbox-builder' )
				);
				wp_send_json_error( $errormessage );
			}

			$nonce = $_POST['nonce'];
			if ( ! wp_verify_nonce( $nonce, 'wdb_tmp_nonce' ) ) {
				$errormessage = array(
					'message' => esc_html__( 'Nonce Varification Faild !', 'designbox-builder' )
				);
				wp_send_json_error( $errormessage );
			}

			$tmpid            = ! empty( $_POST['tmpId'] ) ? sanitize_text_field( $_POST['tmpId'] ) : '';
			$postdata         = get_post( $tmpid );
			$tmpType          = ! empty( get_post_meta( $tmpid, self::CPT_META . '_type', true ) ) ? get_post_meta( $tmpid, self::CPT_META . '_type', true ) : 'single';
			$tmpLocation      = ! empty( get_post_meta( $tmpid, self::CPT_META . '_location', true ) ) ? get_post_meta( $tmpid, self::CPT_META . '_location', true ) : '';
			$specificsDisplay = ! empty( get_post_meta( $tmpid, self::CPT_META . '_splocation', true ) ) ? get_post_meta( $tmpid, self::CPT_META . '_splocation', true ) : '';
			$spLocations      = [];

			if ( ! empty( $specificsDisplay ) ) {
				foreach ( json_decode( $specificsDisplay ) as $item ) {
					$sppost               = get_post( intval( $item ) );
					$spLocations[ $item ] = $sppost->post_title;
				}
			}

			$data = [
				'tmpTitle'      => $postdata->post_title,
				'tmpType'       => $tmpType,
				'tmpLocation'   => $tmpLocation,
				'tmpSpLocation' => $spLocations,
			];
			wp_send_json_success( $data );

		} else {
			$errormessage = array(
				'message' => esc_html__( 'Some thing is worng !', 'designbox-builder' )
			);
			wp_send_json_error( $errormessage );
		}

	}

	/**
	 * Ajax handeler to return the posts based on the search query.
	 * When searching for the post/pages only titles are searched for.
	 *
	 * @since  1.0.0
	 */
	function get_posts_by_query() {

		if ( isset( $_POST ) ) {

			$nonce = $_POST['nonce'];
			if ( ! wp_verify_nonce( $nonce, 'wdb_tmp_nonce' ) ) {
				$errormessage = array(
					'message' => esc_html__( 'Nonce Varification Faild !', 'designbox-builder' )
				);
				wp_send_json_error( $errormessage );
			}

			$search_string = isset( $_POST['q'] ) ? sanitize_text_field( $_POST['q'] ) : '';
			$data          = array();
			$result        = array();

			$args = array(
				'public'   => true,
				'_builtin' => false,
			);

			$output     = 'names'; // names or objects, note names is the default.
			$operator   = 'and'; // also supports 'or'.
			$post_types = get_post_types( $args, $output, $operator );

			unset( $post_types[ self::CPTTYPE ] ); //Exclude wdb post type templates.

			$post_types['Posts'] = 'post';
			$post_types['Pages'] = 'page';

			foreach ( $post_types as $key => $post_type ) {
				$data = array();

				add_filter( 'posts_search', array( $this, 'search_only_titles' ), 10, 2 );

				$query = new \WP_Query(
					array(
						's'              => $search_string,
						'post_type'      => $post_type,
						'posts_per_page' => - 1,
					)
				);

				if ( $query->have_posts() ) {
					while ( $query->have_posts() ) {
						$query->the_post();
						$title  = get_the_title();
						$title  .= ( 0 != $query->post->post_parent ) ? ' (' . get_the_title( $query->post->post_parent ) . ')' : '';
						$id     = get_the_id();
						$data[] = array(
							'id'   => $id,
							'text' => $title,
						);
					}
				}

				if ( is_array( $data ) && ! empty( $data ) ) {
					$result[] = array(
						'text'     => $key,
						'children' => $data,
					);
				}
			}

			$data = array();

			wp_reset_postdata();

			// return the result in json.
			wp_send_json( $result );
		} else {
			$errormessage = array(
				'message' => esc_html__( 'Some thing is worng !', 'designbox-builder' )
			);
			wp_send_json_error( $errormessage );
		}
	}

	/**
	 * Return search results only by post title.
	 * This is only run from hfe_get_posts_by_query()
	 *
	 * @param  (string)   $search   Search SQL for WHERE clause.
	 * @param  (WP_Query) $wp_query The current WP_Query object.
	 *
	 * @return (string) The Modified Search SQL for WHERE clause.
	 */
	function search_only_titles( $search, $wp_query ) {
		if ( ! empty( $search ) && ! empty( $wp_query->query_vars['search_terms'] ) ) {
			global $wpdb;

			$q = $wp_query->query_vars;
			$n = ! empty( $q['exact'] ) ? '' : '%';

			$search = array();

			foreach ( (array) $q['search_terms'] as $term ) {
				$search[] = $wpdb->prepare( "$wpdb->posts.post_title LIKE %s", $n . $wpdb->esc_like( $term ) . $n );
			}

			if ( ! is_user_logged_in() ) {
				$search[] = "$wpdb->posts.post_password = ''";
			}

			$search = ' AND ' . implode( ' AND ', $search );
		}

		return $search;
	}

	/**
	 * Template Insert
	 *
	 * @param [array] $data
	 *
	 * @return void
	 */
	public function insert( $data ) {

		$args        = [
			'post_type'   => self::CPTTYPE,
			'post_status' => $data['tmptype'] == 'popup' ? 'draft' : 'publish',
			'post_title'  => $data['title'],
		];
		$new_post_id = wp_insert_post( $args );

		if ( $new_post_id ) {
			$return = array(
				'message' => esc_html__( 'Template has been inserted', 'designbox-builder' ),
				'id'      => $new_post_id,
			);

			// Meta data
			update_post_meta( $new_post_id, self::CPT_META . '_type', $data['tmptype'] );
			update_post_meta( $new_post_id, self::CPT_META . '_location', $data['tmplocation'] );
			update_post_meta( $new_post_id, '_elementor_edit_mode', 'builder' );
			update_post_meta( $new_post_id, '_wp_page_template', 'elementor_canvas' );

			//specific page and post template header footer
			if ( 'header' === $data['tmptype'] || 'footer' === $data['tmptype'] ) {
				update_post_meta( $new_post_id, self::CPT_META . '_splocation', $data['tmpSpLocation'] );
			}

			wp_send_json_success( $return );

		} else {
			$errormessage = array(
				'message' => esc_html__( 'Some thing is worng !', 'designbox-builder' )
			);
			wp_send_json_error( $errormessage );
		}

	}

	/**
	 * Template Update
	 *
	 * @param [array] $data
	 *
	 * @return void
	 */
	public function update( $data ) {

		$update_post_args = array(
			'ID'         => $data['id'],
			'post_title' => $data['title'],
		);
		wp_update_post( $update_post_args );

		// Update Meta data
		update_post_meta( $data['id'], self::CPT_META . '_type', $data['tmptype'] );
		update_post_meta( $data['id'], self::CPT_META . '_location', $data['tmplocation'] );

		//specific page and post template header footer
		if ( 'header' === $data['tmptype'] || 'footer' === $data['tmptype'] ) {
			update_post_meta( $data['id'], self::CPT_META . '_splocation', $data['tmpSpLocation'] );
		} else {
			delete_post_meta( $data['id'], self::CPT_META . '_splocation' );
		}

		$return = array(
			'message' => esc_html__( 'Template has been updated', 'designbox-builder' ),
			'id'      => $data['id']
		);
		wp_send_json_success( $return );

	}

	/**
	 * Manage Scripts
	 *
	 * @param [string] $hook
	 *
	 * @return void
	 */
	public function enqueue_scripts( $hook ) {

		if ( isset( $_GET['post_type'] ) && $_GET['post_type'] == self::CPTTYPE ) {

			// CSS
			wp_enqueue_style( 'select2', WDB_ADDONS_URL . '/assets/css/select2.min.css' );
			wp_enqueue_style( 'wdb-theme-builder', WDB_ADDONS_URL . '/assets/css/theme-builder.css' );

			// JS
			wp_enqueue_script( 'select2', WDB_ADDONS_URL . '/assets/js/select2.min.js', array( 'jquery' ), WDB_ADDONS_VERSION, true );
			wp_enqueue_script( 'wdb-theme-builder', WDB_ADDONS_URL . '/assets/js/theme-builder.js', array(
				'jquery',
				'wp-util'
			), WDB_ADDONS_VERSION, true );

			$localize_data = [
				'ajaxurl'         => admin_url( 'admin-ajax.php' ),
				'nonce'           => wp_create_nonce( 'wdb_tmp_nonce' ),
				'adminURL'        => admin_url(),
				'hflocation'      => self::get_hf_location_selections(),
				'archivelocation' => self::get_archive_location_selections(),
				'singlelocation'  => self::get_single_location_selections(),
				'templatetype'    => self::get_template_type(),
				'labels'          => [
					'fields'  => [
						'name'    => [
							'title'       => esc_html__( 'Name', 'designbox-builder' ),
							'placeholder' => esc_html__( 'Enter a template name', 'designbox-builder' )
						],
						'type'    => esc_html__( 'Type', 'designbox-builder' ),
						'display' => esc_html__( 'Display', 'designbox-builder' ),
					],
					'head'    => esc_html__( 'Template Settings', 'designbox-builder' ),
					'buttons' => [
						'elementor' => [
							'label' => esc_html__( 'Edit With Elementor', 'designbox-builder' ),
							'link'  => '#'
						],
						'save'      => [
							'label'  => esc_html__( 'Save Settings', 'designbox-builder' ),
							'saving' => esc_html__( 'Saving...', 'designbox-builder' ),
							'saved'  => esc_html__( 'All Data Saved', 'designbox-builder' ),
							'link'   => '#'
						]
					],
				]
			];
			wp_localize_script( 'wdb-theme-builder', 'WDB_Theme_Builder', $localize_data );

		}

	}

	/**
	 * [init] Assets Initializes
	 * @return [void]
	 */
	public function init() {
		//Register Custom Post Type
		$this->register_custom_post_type();
	}

	/**
	 * [admin_menu] Add Post type Submenu
	 *
	 * @return void
	 */
	public function admin_menu() {
		$link_custom_post = 'edit.php?post_type=' . self::CPTTYPE;
		add_submenu_page(
			'wdb_addons_page',
			esc_html__( 'Theme Builder', 'designbox-builder' ),
			esc_html__( 'Theme Builder', 'designbox-builder' ),
			'manage_options',
			$link_custom_post,
			null
		);
	}

}

WDB_Theme_Builder::instance();
