<?php
// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Ensure is_plugin_active() is available
if ( ! function_exists( 'is_plugin_active' ) ) {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
}


// Rules JSON URL
if ( ! defined( 'KEYSTONE_LICENSE_RULES_URL' ) ) {
    define( 'KEYSTONE_LICENSE_RULES_URL', BUCKETAPI_URL . '/wp-content/uploads/bucket-manager/license-rules.json' );
}


/**
 * Deactivate conflicting plugins and show admin notice once per plugin.
 */
function keystone_deactivate_conflicting_plugins() {
    $plugins_to_deactivate = [
        'keystone-essential/keystone-essential.php' => __( 'The KeyStone Essential plugin has been deactivated because Keystone Framework is now active.', 'keystone-framework' ),
        'keystone-lite-framework/keystone-lite-framework.php' => __( 'The Keystone Lite Framework plugin has been deactivated because Keystone Framework is now active.', 'keystone-framework' ),
    ];

    $notices = [];

    foreach ( $plugins_to_deactivate as $plugin => $message ) {
        if ( is_plugin_active( $plugin ) ) {
            deactivate_plugins( $plugin );
            $notices[] = $message;
        }
    }

    if ( ! empty( $notices ) ) {
        add_action( 'admin_notices', function() use ( $notices ) {
            foreach ( $notices as $message ) {
                printf(
                    '<div class="notice notice-warning is-dismissible"><p>%s</p></div>',
                    esc_html( $message )
                );
            }
        });
    }
}
add_action( 'admin_init', 'keystone_deactivate_conflicting_plugins' );

/**
 * Fetch and cache license rules from remote JSON.
 *
 * @return array
 */
function keystone_get_license_rules(): array {
    $cache_key = 'keystone_license_rules';

    // Set this to true to bypass cache for testing
    $bypass_cache = false;

    if ( ! $bypass_cache ) {
        $rules = get_transient( $cache_key );
        if ( false !== $rules ) {
            return $rules;
        }
    }

    $response = wp_remote_get( KEYSTONE_LICENSE_RULES_URL, [ 'timeout' => 10 ] );

    if ( is_wp_error( $response ) ) {
        return [];
    }

    $body = wp_remote_retrieve_body( $response );
    $decoded = json_decode( $body, true );

    if ( is_array( $decoded ) && isset( $decoded['rules'] ) && is_array( $decoded['rules'] ) ) {
        $rules = $decoded['rules'];
        if ( ! $bypass_cache ) {
            set_transient( $cache_key, $rules, HOUR_IN_SECONDS );
        }
        return $rules;
    }

    return [];
}


/**
 * Check if current theme requires license.
 *
 * @return bool
 */
function keystone_theme_requires_license(): bool {
    static $slug = null;
    if ( $slug === null ) {
        $slug = wp_get_theme()->get_stylesheet();
    }

    $rules = keystone_get_license_rules();
    return $rules[ $slug ] ?? false;
}

/**
 * Display remote message notice in admin area.
 */
function keystone_remote_insta_notice() {
    $user_id = get_current_user_id();
    if ( ! $user_id ) {
        return;
    }

    $dismissed = get_user_meta( $user_id, 'remote_insta_notice_dismissed_at', true );
    if ( $dismissed && ( time() - intval( $dismissed ) < 3 * DAY_IN_SECONDS ) ) {
        return;
    }

    $transient_key = 'remote_insta_notice_data';
    $data = get_transient( $transient_key );

    if ( ! is_array( $data ) ) {
        $response = wp_remote_get( KEYSTONE_LICENSE_RULES_URL, [ 'timeout' => 5 ] );

        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            $data = json_decode( wp_remote_retrieve_body( $response ), true );
            if ( is_array( $data ) ) {
                set_transient( $transient_key, $data, 12 * HOUR_IN_SECONDS );
            } else {
                $data = [];
            }
        }
    }

    static $slug = null;
    if ( $slug === null ) {
        $slug = wp_get_theme()->get_stylesheet();
    }

    if ( empty( $data['theme_info'][ $slug ]['show'] ) || empty( $data['theme_info'][ $slug ]['message'] ) ) {
        return;
    }

    $notice = $data['theme_info'][ $slug ];
    $type = in_array( $notice['type'], ['warning','error','success','info'], true ) ? $notice['type'] : 'info';

    ?>
    <div id="insta-notice-container">
        <div id="insta-notice" class="notice notice-<?php echo esc_attr( $type ); ?> is-dismissible">
            <p><?php echo wp_kses_post( $notice['message'] ); ?></p>
        </div>
    </div>
    <script>
    (function($){
        $(document).on('click', '#insta-notice .notice-dismiss', function(){
            $.post(ajaxurl, { action: 'remote_insta_notice_dismiss' });
        });
    })(jQuery);
    </script>
    <?php
}
add_action( 'admin_notices', 'keystone_remote_insta_notice' );

/**
 * Handle AJAX dismiss of insta notice.
 */
function keystone_remote_insta_notice_dismiss() {
    $user_id = get_current_user_id();
    if ( $user_id ) {
        update_user_meta( $user_id, 'remote_insta_notice_dismissed_at', time() );
    }
    wp_die();
}
add_action( 'wp_ajax_remote_insta_notice_dismiss', 'keystone_remote_insta_notice_dismiss' );

