<?php
/**
 * Class for the customizer importer used in the Smart One Click Setup plugin.
 *
 * Code is mostly from the Customizer Export/Import plugin.
 *
 * @see https://wordpress.org/plugins/customizer-export-import/
 * @package socs
 */

namespace SOCS;

class CustomizerImporter {
	/**
	 * Import customizer from a DAT file, generated by the Customizer Export/Import plugin.
	 *
	 * @param string $customizer_import_file_path path to the customizer import file.
	 */
	public static function import( $customizer_import_file_path ) {
		$socs          = SmartOneClickSetup::get_instance();
		$log_file_path = $socs->get_log_file_path();

		// Try to import the customizer settings.
		$results = self::import_customizer_options( $customizer_import_file_path );

		// Check for errors, else write the results to the log file.
		if ( is_wp_error( $results ) ) {
			$error_message = $results->get_error_message();

			// Add any error messages to the frontend_error_messages variable in SOCS main class.
			$socs->append_to_frontend_error_messages( $error_message );

			// Write error to log file.
			Helpers::append_to_file(
				$error_message,
				$log_file_path,
				esc_html__( 'Importing customizer settings', 'smart-one-click-setup' )
			);
		}
		else {
			// Add this message to log file.
			$log_added = Helpers::append_to_file(
				esc_html__( 'Customizer settings import finished!', 'smart-one-click-setup' ),
				$log_file_path,
				esc_html__( 'Importing customizer settings' , 'smart-one-click-setup' )
			);
		}
	}


	/**
	 * Imports uploaded mods and calls WordPress core customize_save actions so
	 * themes that hook into them can act before mods are saved to the database.
	 *
	 * Update: WP core customize_save actions were removed, because of some errors.
	 *
	 * @since 1.1.1
	 * @param string $import_file_path Path to the import file.
	 * @return void|WP_Error
	 */
	public static function import_customizer_options( $import_file_path ) {
		// Setup global vars.
		global $wp_customize;

		// Setup internal vars.
		$template = get_template();
		$stylesheet = get_stylesheet();

		// Make sure we have an import file.
		if ( ! file_exists( $import_file_path ) ) {
			return new \WP_Error(
				'missing_cutomizer_import_file',
				sprintf( /* translators: %s - file path */
					esc_html__( 'Error: The customizer import file is missing! File path: %s', 'smart-one-click-setup' ),
					$import_file_path
				)
			);
		}

		// Get the upload data.
		$raw = Helpers::data_from_file( $import_file_path );

		// Make sure we got the data.
		if ( is_wp_error( $raw ) ) {
			return $raw;
		}

		$data = unserialize( $raw , array( 'allowed_classes' => false ) );

		// Data checks - support both 'theme' and 'template' keys for backward compatibility.
		if ( ! is_array( $data ) || ( ! isset( $data['theme'] ) && ! isset( $data['template'] ) ) || ! isset( $data['mods'] ) ) {
			return new \WP_Error(
				'customizer_import_data_error',
				esc_html__( 'Error: The customizer import file is not in a correct format. Please make sure to use the correct customizer import file.', 'smart-one-click-setup' )
			);
		}

		// Get the theme from export data (support both 'theme' and 'template' keys).
		$export_theme = isset( $data['theme'] ) ? $data['theme'] : ( isset( $data['template'] ) ? $data['template'] : '' );

		// Check if theme matches (allow same theme or child theme).
		// If export theme matches template, allow import (same parent theme or same theme).
		// If export theme matches stylesheet, allow import (same theme).
		// This allows importing customizer settings for the same theme or parent theme.
		if ( empty( $export_theme ) || ( $export_theme !== $template && $export_theme !== $stylesheet ) ) {
			return new \WP_Error(
				'customizer_import_wrong_theme',
				esc_html__( 'Error: The customizer import file is not suitable for current theme. You can only import customizer settings for the same theme or a child theme.', 'smart-one-click-setup' )
			);
		}

		// Import images.
		if ( Helpers::apply_filters( 'socs/customizer_import_images', true ) ) {
			$data['mods'] = self::import_customizer_images( $data['mods'] );
		}

		// Import custom options.
		if ( isset( $data['options'] ) ) {
			// Require modified customizer options class.
			if ( ! class_exists( '\WP_Customize_Setting' ) ) {
				require_once ABSPATH . 'wp-includes/class-wp-customize-setting.php';
			}

			// Get post ID mapping from content import (for remapping page IDs).
			$post_id_mapping = array();
			$socs = SmartOneClickSetup::get_instance();
			
			// Try to get mapping from importer instance first.
			if ( ! empty( $socs->importer ) && method_exists( $socs->importer, 'get_importer_data' ) ) {
				$importer_data = $socs->importer->get_importer_data();
				if ( ! empty( $importer_data['mapping']['post'] ) ) {
					// Mapping format: array( 'old_id' => 'new_id' ).
					$post_id_mapping = $importer_data['mapping']['post'];
				}
			}
			
			// If mapping not available from importer, try to get it from transient.
			// This is important for customizer import which happens in a separate AJAX call.
			if ( empty( $post_id_mapping ) ) {
				$transient_data = get_transient( 'socs_importer_data' );
				if ( ! empty( $transient_data ) && ! empty( $transient_data['mapping']['post'] ) ) {
					$post_id_mapping = $transient_data['mapping']['post'];
				}
			}

			// Process show_on_front first, then page_on_front and page_for_posts.
			// This ensures show_on_front is set before page_on_front.
			$show_on_front = null;
			$page_on_front = null;
			$page_for_posts = null;
			
			foreach ( $data['options'] as $option_key => $option_value ) {
				if ( 'show_on_front' === $option_key ) {
					$show_on_front = $option_value;
				} elseif ( 'page_on_front' === $option_key ) {
					$page_on_front = $option_value;
				} elseif ( 'page_for_posts' === $option_key ) {
					$page_for_posts = $option_value;
				}
			}
			
			// Import all other options first (non-home-page options).
			foreach ( $data['options'] as $option_key => $option_value ) {
				// Skip home page settings - we'll handle them separately.
				if ( in_array( $option_key, array( 'show_on_front', 'page_on_front', 'page_for_posts' ), true ) ) {
					continue;
				}

				$option = new CustomizerOption( $wp_customize, $option_key, array(
					'default'    => '',
					'type'       => 'option',
					'capability' => 'edit_theme_options',
				) );

				$option->import( $option_value );
			}
			
			// Now handle home page settings with proper remapping and validation.
			// First, set show_on_front if it exists.
			if ( null !== $show_on_front ) {
				update_option( 'show_on_front', $show_on_front );
			}
			
			// Remap and set page_on_front if needed.
			if ( null !== $page_on_front && ! empty( $page_on_front ) ) {
				$mapped_page_id = $page_on_front;
				
				// Check if we have a mapping for this page ID.
				if ( ! empty( $post_id_mapping[ $page_on_front ] ) ) {
					$mapped_page_id = $post_id_mapping[ $page_on_front ];
				}
				
				// Verify the page exists and is a valid page.
				$page_exists = get_post( $mapped_page_id );
				if ( $page_exists && 'page' === $page_exists->post_type ) {
					update_option( 'page_on_front', $mapped_page_id );
					// Ensure show_on_front is set to 'page' if page_on_front is set.
					if ( null === $show_on_front || 'page' !== $show_on_front ) {
						update_option( 'show_on_front', 'page' );
					}
				} else {
					// Log warning if page not found.
					$socs = SmartOneClickSetup::get_instance();
					$log_file_path = $socs->get_log_file_path();
					Helpers::append_to_file(
						sprintf(
							/* translators: %s: Page ID */
							esc_html__( 'Warning: Home page with ID %s was not found or is not a valid page. Home page setting was not applied.', 'smart-one-click-setup' ),
							$mapped_page_id
						),
						$log_file_path,
						esc_html__( 'Importing customizer settings', 'smart-one-click-setup' )
					);
				}
			}
			
			// Remap and set page_for_posts if needed.
			if ( null !== $page_for_posts && ! empty( $page_for_posts ) ) {
				$mapped_page_id = $page_for_posts;
				
				// Check if we have a mapping for this page ID.
				if ( ! empty( $post_id_mapping[ $page_for_posts ] ) ) {
					$mapped_page_id = $post_id_mapping[ $page_for_posts ];
				}
				
				// Verify the page exists and is a valid page.
				$page_exists = get_post( $mapped_page_id );
				if ( $page_exists && 'page' === $page_exists->post_type ) {
					update_option( 'page_for_posts', $mapped_page_id );
				} else {
					// Log warning if page not found.
					$socs = SmartOneClickSetup::get_instance();
					$log_file_path = $socs->get_log_file_path();
					Helpers::append_to_file(
						sprintf(
							/* translators: %s: Page ID */
							esc_html__( 'Warning: Posts page with ID %s was not found or is not a valid page. Posts page setting was not applied.', 'smart-one-click-setup' ),
							$mapped_page_id
						),
						$log_file_path,
						esc_html__( 'Importing customizer settings', 'smart-one-click-setup' )
					);
				}
			}
		}

		// Should the customizer import use the WP customize_save* hooks?
		$use_wp_customize_save_hooks = Helpers::apply_filters( 'socs/enable_wp_customize_save_hooks', false );

		if ( $use_wp_customize_save_hooks ) {
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			do_action( 'customize_save', $wp_customize );
		}

		// Loop through the mods and save the mods.
		foreach ( $data['mods'] as $key => $val ) {
			if ( $use_wp_customize_save_hooks ) {
				// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
				do_action( 'customize_save_' . $key, $wp_customize );
			}

			set_theme_mod( $key, $val );
		}

		if ( $use_wp_customize_save_hooks ) {
			// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
			do_action( 'customize_save_after', $wp_customize );
		}
	}

	/**
	 * Helper function: Customizer import - imports images for settings saved as mods.
	 *
	 * @since 1.1.1
	 * @param array $mods An array of customizer mods.
	 * @return array The mods array with any new import data.
	 */
	private static function import_customizer_images( $mods ) {
		foreach ( $mods as $key => $val ) {
			if ( self::customizer_is_image_url( $val ) ) {
				$data = self::customizer_sideload_image( $val );
				if ( ! is_wp_error( $data ) ) {
					$mods[ $key ] = $data->url;

					// Handle header image controls.
					if ( isset( $mods[ $key . '_data' ] ) ) {
						$mods[ $key . '_data' ] = $data;
						update_post_meta( $data->attachment_id, '_wp_attachment_is_custom_header', get_stylesheet() );
					}
				}
			}
		}

		return $mods;
	}

	/**
	 * Helper function: Customizer import
	 * Taken from the core media_sideload_image function and
	 * modified to return an array of data instead of html.
	 *
	 * @since 1.1.1.
	 * @param string $file The image file path.
	 * @return array An array of image data.
	 */
	private static function customizer_sideload_image( $file ) {
		$data = new \stdClass();

		if ( ! function_exists( 'media_handle_sideload' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/media.php' );
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
			require_once( ABSPATH . 'wp-admin/includes/image.php' );
		}
		if ( ! empty( $file ) ) {
			// Set variables for storage, fix file filename for query strings.
			preg_match( '/[^\?]+\.(jpe?g|jpe|gif|png)\b/i', $file, $matches );
			$file_array = array();
			$file_array['name'] = basename( $matches[0] );

			// Download file to temp location.
			$file_array['tmp_name'] = download_url( $file );

			// If error storing temporarily, return the error.
			if ( is_wp_error( $file_array['tmp_name'] ) ) {
				return $file_array['tmp_name'];
			}

			// Do the validation and storage stuff.
			$id = media_handle_sideload( $file_array, 0 );

			// If error storing permanently, delete temporary file.
			if ( is_wp_error( $id ) ) {
				wp_delete_file( $file_array['tmp_name'] );
				return $id;
			}

			// Build the object to return.
			$meta                = wp_get_attachment_metadata( $id );
			$data->attachment_id = $id;
			$data->url           = wp_get_attachment_url( $id );
			$data->thumbnail_url = wp_get_attachment_thumb_url( $id );
			$data->height        = $meta['height'];
			$data->width         = $meta['width'];
		}

		return $data;
	}

	/**
	 * Checks to see whether a string is an image url or not.
	 *
	 * @since 1.1.1
	 * @param string $string The string to check.
	 * @return bool Whether the string is an image url or not.
	 */
	private static function customizer_is_image_url( $string = '' ) {
		if ( is_string( $string ) ) {
			if ( preg_match( '/\.(jpg|jpeg|png|gif)/i', $string ) ) {
				return true;
			}
		}

		return false;
	}
}
