<?php
/**
 * Class for the import actions used in the Smart One Click Setup plugin.
 * Register default WP actions for SOCS plugin.
 *
 * @package socs
 */

namespace SOCS;

class ImportActions {
	/**
	 * Register all action hooks for this class.
	 */
	public function register_hooks() {
		// Before content import.
		add_action( 'socs/before_content_import_execution', array( $this, 'before_content_import_action' ), 10, 3 );

		// After content import.
		add_action( 'socs/after_content_import_execution', array( $this, 'before_widget_import_action' ), 10, 3 );
		add_action( 'socs/after_content_import_execution', array( $this, 'widgets_import' ), 20, 3 );
		add_action( 'socs/after_content_import_execution', array( $this, 'redux_import' ), 30, 3 );
		add_action( 'socs/after_content_import_execution', array( $this, 'wpforms_import' ), 40, 3 );
		add_action( 'socs/after_content_import_execution', array( $this, 'elementor_import' ), 50, 3 );
		add_action( 'socs/after_content_import_execution', array( $this, 'plugin_settings_import' ), 60, 3 );

		// Customizer import.
		add_action( 'socs/customizer_import_execution', array( $this, 'customizer_import' ), 10, 1 );

		// After full import action.
		add_action( 'socs/after_all_import_execution', array( $this, 'after_import_action' ), 10, 3 );
		add_action( 'socs/after_all_import_execution', array( $this, 'clear_elementor_cache' ), 20, 3 );

		// Special widget import cases.
		if ( Helpers::apply_filters( 'socs/enable_custom_menu_widget_ids_fix', true ) ) {
			add_action( 'socs/widget_settings_array', array( $this, 'fix_custom_menu_widget_ids' ) );
		}
	}


	/**
	 * Change the menu IDs in the custom menu widgets in the widget import data.
	 * This solves the issue with custom menu widgets not having the correct (new) menu ID, because they
	 * have the old menu ID from the export site.
	 *
	 * @since 3.4.0 Made sure `$widget` is an array.
	 *
	 * @param array $widget The widget settings array.
	 */
	public function fix_custom_menu_widget_ids( $widget ) {

		// Make sure the passed variable is an array to prevent fatal error.
		if ( ! is_array( $widget ) ) {
			$widget = [];
		}

		// Skip (no changes needed), if this is not a custom menu widget.
		if ( ! array_key_exists( 'nav_menu', $widget ) || empty( $widget['nav_menu'] ) || ! is_int( $widget['nav_menu'] ) ) {
			return $widget;
		}

		// Get import data, with new menu IDs.
		$socs                = SmartOneClickSetup::get_instance();
		$content_import_data = $socs->importer->get_importer_data();
		$term_ids            = $content_import_data['mapping']['term_id'];

		// Set the new menu ID for the widget.
		$widget['nav_menu'] = $term_ids[ $widget['nav_menu'] ];

		return $widget;
	}


	/**
	 * Execute the widgets import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function widgets_import( $selected_import_files, $import_files, $selected_index ) {
		if ( ! empty( $selected_import_files['widgets'] ) ) {
			WidgetImporter::import( $selected_import_files['widgets'] );
		}
	}


	/**
	 * Execute the Redux import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function redux_import( $selected_import_files, $import_files, $selected_index ) {
		if ( ! empty( $selected_import_files['redux'] ) ) {
			ReduxImporter::import( $selected_import_files['redux'] );
		}
	}

	/**
	 * Execute the WPForms import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function wpforms_import( $selected_import_files, $import_files, $selected_index ) {
		if ( ! empty( $selected_import_files['wpforms'] ) ) {
			( new WPFormsImporter( $selected_import_files['wpforms'] ) )->import();
		}
	}

	/**
	 * Execute the Elementor import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, elementor).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function elementor_import( $selected_import_files, $import_files, $selected_index ) {
		if ( ! empty( $selected_import_files['elementor'] ) ) {
			ElementorImporter::import( $selected_import_files['elementor'] );
		}
	}

	/**
	 * Execute the plugin settings import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, plugins).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function plugin_settings_import( $selected_import_files, $import_files, $selected_index ) {
		if ( ! empty( $selected_import_files['plugins'] ) ) {
			PluginSettingsImporter::import( $selected_import_files['plugins'] );
		}
	}

	/**
	 * Execute the customizer import.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function customizer_import( $selected_import_files ) {
		if ( ! empty( $selected_import_files['customizer'] ) ) {
			CustomizerImporter::import( $selected_import_files['customizer'] );
		}
	}


	/**
	 * Execute the action: 'socs/before_content_import'.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function before_content_import_action( $selected_import_files, $import_files, $selected_index ) {
		$this->do_import_action( 'socs/before_content_import', $import_files[ $selected_index ] );
	}


	/**
	 * Execute the action: 'socs/before_widgets_import'.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function before_widget_import_action( $selected_import_files, $import_files, $selected_index ) {
		$this->do_import_action( 'socs/before_widgets_import', $import_files[ $selected_index ] );
	}


	/**
	 * Execute the action: 'socs/after_import'.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function after_import_action( $selected_import_files, $import_files, $selected_index ) {
		$this->do_import_action( 'socs/after_import', $import_files[ $selected_index ] );
	}

	/**
	 * Clear Elementor cache after all imports complete.
	 * This ensures Elementor widgets (especially list widgets on blog/WooCommerce pages) display correctly.
	 *
	 * @param array $selected_import_files Actual selected import files (content, widgets, customizer, redux).
	 * @param array $import_files          The filtered import files defined in `socs/import_files` filter.
	 * @param int   $selected_index        Selected index of import.
	 */
	public function clear_elementor_cache( $selected_import_files, $import_files, $selected_index ) {
		// Check if Elementor is active.
		if ( ! class_exists( '\Elementor\Plugin' ) ) {
			return;
		}

		$elementor = \Elementor\Plugin::$instance;

		// Clear Elementor files cache.
		if ( method_exists( $elementor->files_manager, 'clear_cache' ) ) {
			$elementor->files_manager->clear_cache();
		}

		// Clear Elementor posts CSS cache (important for widgets to display correctly).
		if ( method_exists( $elementor->posts_css_manager, 'clear_cache' ) ) {
			$elementor->posts_css_manager->clear_cache();
		}

		// Clear Elementor kits cache.
		if ( method_exists( $elementor->kits_manager, 'clear_cache' ) ) {
			$elementor->kits_manager->clear_cache();
		}

		// Also try to clear Elementor's general cache if available.
		if ( method_exists( $elementor, 'files_manager' ) && method_exists( $elementor->files_manager, 'regenerate_css_files' ) ) {
			$elementor->files_manager->regenerate_css_files();
		}
	}


	/**
	 * Register the do_action hook, so users can hook to these during import.
	 *
	 * @param string $action          The action name to be executed.
	 * @param array  $selected_import The data of selected import from `socs/import_files` filter.
	 */
	private function do_import_action( $action, $selected_import ) {
		if ( false !== Helpers::has_action( $action ) ) {
			$socs          = SmartOneClickSetup::get_instance();
			$log_file_path = $socs->get_log_file_path();

			ob_start();
				Helpers::do_action( $action, $selected_import );
			$message = ob_get_clean();

			// Add this message to log file.
			$log_added = Helpers::append_to_file(
				$message,
				$log_file_path,
				$action
			);
		}
	}
}
