<?php
/**
 * Class for the Redux importer used in the Smart One Click Setup plugin.
 *
 * @see https://wordpress.org/plugins/redux-framework/
 * @package socs
 */

namespace SOCS;

class ReduxImporter {
	/**
	 * Import Redux data from a JSON file, generated by the Redux plugin.
	 *
	 * @param array $import_data Array of arrays. Child array contains 'option_name' and 'file_path'.
	 */
	public static function import( $import_data ) {
		$socs          = SmartOneClickSetup::get_instance();
		$log_file_path = $socs->get_log_file_path();

		// Redux plugin is not active!
		if ( ! class_exists( 'ReduxFramework' ) ) {
			$error_message = esc_html__( 'The Redux plugin is not activated, so the Redux import was skipped!', 'smart-one-click-setup' );

			// Add any error messages to the frontend_error_messages variable in SOCS main class.
			$socs->append_to_frontend_error_messages( $error_message );

			// Write error to log file.
			Helpers::append_to_file(
				$error_message,
				$log_file_path,
				esc_html__( 'Importing Redux settings' , 'smart-one-click-setup' )
			);

			return;
		}

		foreach ( $import_data as $redux_item ) {
			$redux_options_raw_data = Helpers::data_from_file( $redux_item['file_path'] );

			$redux_options_data = json_decode( $redux_options_raw_data, true );

			$redux_framework = \ReduxFrameworkInstances::get_instance( $redux_item['option_name'] );

			if ( isset( $redux_framework->args['opt_name'] ) ) {
				// Import Redux settings.
				if ( ! empty( $redux_framework->options_class ) && method_exists( $redux_framework->options_class, 'set' ) ) {
					$redux_framework->options_class->set( $redux_options_data );
				} else {
					// Handle backwards compatibility.
					$redux_framework->set_options( $redux_options_data );
				}

				// Add this message to log file.
				$log_added = Helpers::append_to_file( /* translators: %s - the name of the Redux option. */
					sprintf( esc_html__( 'Redux settings import for: %s finished successfully!', 'smart-one-click-setup' ), $redux_item['option_name'] ),
					$log_file_path,
					esc_html__( 'Importing Redux settings' , 'smart-one-click-setup' )
				);
			}
			else { /* translators: %s - the name of the Redux option. */
				$error_message = sprintf( esc_html__( 'The Redux option name: %s, was not found in this WP site, so it was not imported!', 'smart-one-click-setup' ), $redux_item['option_name'] );

				// Add any error messages to the frontend_error_messages variable in SOCS main class.
				$socs->append_to_frontend_error_messages( $error_message );

				// Write error to log file.
				Helpers::append_to_file(
					$error_message,
					$log_file_path,
					esc_html__( 'Importing Redux settings' , 'smart-one-click-setup' )
				);
			}
		}
	}
}
