<?php
namespace KeyStone\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Plugin_Details {
	
	public function __construct() {
		add_filter( 'plugin_row_meta', [ $this, 'add_view_details_link' ], 10, 2 );
		add_action( 'admin_footer', [ $this, 'add_plugin_details_modal' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
	}

	/**
	 * Add "View details" link to plugin row
	 */
	public function add_view_details_link( $links, $file ) {
		if ( KEYSTONE_PLUGIN_BASE === $file ) {
			$view_details = '<a href="#" class="keystone-view-details" data-plugin="keystone-framework">' . esc_html__( 'View details', 'keystone-framework' ) . '</a>';
			$links[] = $view_details;
		}
		return $links;
	}

	/**
	 * Get readme content and convert to HTML
	 */
	private function get_readme_content() {
		$readme_file = KEYSTONE_PATH . 'readme.md';
		
		if ( ! file_exists( $readme_file ) ) {
			return '<p>' . esc_html__( 'Readme file not found.', 'keystone-framework' ) . '</p>';
		}

		$content = file_get_contents( $readme_file );
		
		// Simple markdown to HTML conversion
		$content = $this->markdown_to_html( $content );
		
		return $content;
	}

	/**
	 * Simple markdown to HTML converter
	 */
	private function markdown_to_html( $text ) {
		// Convert headers
		$text = preg_replace( '/^### (.*?)$/m', '<h3>$1</h3>', $text );
		$text = preg_replace( '/^## (.*?)$/m', '<h2>$1</h2>', $text );
		$text = preg_replace( '/^# (.*?)$/m', '<h1>$1</h1>', $text );
		
		// Convert bold
		$text = preg_replace( '/\*\*(.*?)\*\*/', '<strong>$1</strong>', $text );
		
		// Convert code blocks
		$text = preg_replace( '/`([^`]+)`/', '<code>$1</code>', $text );
		
		// Convert unordered lists
		$text = preg_replace_callback( '/^(\s*)- (.*)$/m', function( $matches ) {
			$indent = strlen( $matches[1] );
			$level = $indent / 2;
			return str_repeat( '  ', $level ) . '<li>' . $matches[2] . '</li>';
		}, $text );
		
		// Wrap lists in ul tags
		$text = preg_replace( '/(<li>.*<\/li>\n?)+/s', '<ul>$0</ul>', $text );
		
		// Convert links [text](url)
		$text = preg_replace( '/\[([^\]]+)\]\(([^\)]+)\)/', '<a href="$2" target="_blank">$1</a>', $text );
		
		// Convert line breaks to paragraphs
		$paragraphs = explode( "\n\n", $text );
		$formatted = '';
		foreach ( $paragraphs as $paragraph ) {
			$paragraph = trim( $paragraph );
			if ( ! empty( $paragraph ) && ! preg_match( '/^<(h[1-6]|ul|ol|li|code)/', $paragraph ) ) {
				$formatted .= '<p>' . $paragraph . '</p>';
			} else {
				$formatted .= $paragraph;
			}
		}
		
		return $formatted;
	}

	/**
	 * Add modal HTML to admin footer
	 */
	public function add_plugin_details_modal() {
		$screen = get_current_screen();
		
		// Only show on plugins page
		if ( 'plugins' !== $screen->id ) {
			return;
		}
		
		$readme_content = $this->get_readme_content();
		?>
		<div id="keystone-plugin-details-modal" class="keystone-modal" style="display: none;">
			<div class="keystone-modal-overlay"></div>
			<div class="keystone-modal-content">
				<div class="keystone-modal-header">
					<h2><?php echo esc_html__( 'Keystone Framework - Plugin Details', 'keystone-framework' ); ?></h2>
					<button class="keystone-modal-close" aria-label="<?php esc_attr_e( 'Close', 'keystone-framework' ); ?>">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div class="keystone-modal-body">
					<?php echo wp_kses_post( $readme_content ); ?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Enqueue scripts and styles for the modal
	 */
	public function enqueue_scripts( $hook ) {
		if ( 'plugins.php' !== $hook ) {
			return;
		}

		// Add inline styles
		wp_add_inline_style( 'keystone-admin', $this->get_modal_styles() );
		
		// Add inline script
		wp_add_inline_script( 'jquery', $this->get_modal_script() );
	}

	/**
	 * Get modal CSS
	 */
	private function get_modal_styles() {
		return '
		.keystone-modal {
			position: fixed;
			top: 0;
			left: 0;
			width: 100%;
			height: 100%;
			z-index: 100000;
		}
		
		.keystone-modal-overlay {
			position: absolute;
			top: 0;
			left: 0;
			width: 100%;
			height: 100%;
			background: rgba(0, 0, 0, 0.7);
		}
		
		.keystone-modal-content {
			position: relative;
			max-width: 900px;
			max-height: 90vh;
			margin: 5vh auto;
			background: #fff;
			border-radius: 4px;
			box-shadow: 0 5px 30px rgba(0, 0, 0, 0.3);
			display: flex;
			flex-direction: column;
		}
		
		.keystone-modal-header {
			padding: 20px 30px;
			border-bottom: 1px solid #ddd;
			display: flex;
			justify-content: space-between;
			align-items: center;
		}
		
		.keystone-modal-header h2 {
			margin: 0;
			font-size: 20px;
			font-weight: 600;
		}
		
		.keystone-modal-close {
			background: none;
			border: none;
			cursor: pointer;
			padding: 0;
			font-size: 24px;
			color: #666;
			line-height: 1;
		}
		
		.keystone-modal-close:hover {
			color: #000;
		}
		
		.keystone-modal-close .dashicons {
			width: 24px;
			height: 24px;
			font-size: 24px;
		}
		
		.keystone-modal-body {
			padding: 30px;
			overflow-y: auto;
			flex: 1;
		}
		
		.keystone-modal-body h1 {
			font-size: 26px;
			margin-top: 0;
			margin-bottom: 15px;
		}
		
		.keystone-modal-body h2 {
			font-size: 22px;
			margin-top: 30px;
			margin-bottom: 15px;
			border-bottom: 1px solid #ddd;
			padding-bottom: 10px;
		}
		
		.keystone-modal-body h3 {
			font-size: 18px;
			margin-top: 25px;
			margin-bottom: 10px;
		}
		
		.keystone-modal-body ul {
			margin-left: 20px;
			margin-bottom: 15px;
		}
		
		.keystone-modal-body li {
			margin-bottom: 8px;
			line-height: 1.6;
		}
		
		.keystone-modal-body p {
			line-height: 1.6;
			margin-bottom: 15px;
		}
		
		.keystone-modal-body code {
			background: #f5f5f5;
			padding: 2px 6px;
			border-radius: 3px;
			font-family: monospace;
			font-size: 13px;
		}
		
		.keystone-modal-body strong {
			font-weight: 600;
		}
		
		.keystone-modal-body a {
			color: #2271b1;
			text-decoration: none;
		}
		
		.keystone-modal-body a:hover {
			text-decoration: underline;
		}
		';
	}

	/**
	 * Get modal JavaScript
	 */
	private function get_modal_script() {
		return '
		jQuery(document).ready(function($) {
			// Open modal
			$(document).on("click", ".keystone-view-details", function(e) {
				e.preventDefault();
				$("#keystone-plugin-details-modal").fadeIn(200);
				$("body").css("overflow", "hidden");
			});
			
			// Close modal
			$(document).on("click", ".keystone-modal-close, .keystone-modal-overlay", function(e) {
				e.preventDefault();
				$("#keystone-plugin-details-modal").fadeOut(200);
				$("body").css("overflow", "auto");
			});
			
			// Close on Escape key
			$(document).on("keydown", function(e) {
				if (e.keyCode === 27 && $("#keystone-plugin-details-modal").is(":visible")) {
					$("#keystone-plugin-details-modal").fadeOut(200);
					$("body").css("overflow", "auto");
				}
			});
		});
		';
	}
}

new Plugin_Details();

